/*
 * Written by Oron Peled <oron@actcom.co.il>
 * Copyright (C) 2004-2006, Xorcom
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <linux/version.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#  warning "This module is tested only with 2.6 kernels"
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,12)
#  undef USB_FIELDS_MISSING
#else
#  define USB_FIELDS_MISSING

#  define USB_MAX_STRING	128
#  define USB_GET_STRING(udev,field,buf)		\
	do {						\
		if((udev)->descriptor.field) {		\
			char	tmp[USB_MAX_STRING];	\
			if(usb_string((udev), (udev)->descriptor.field, tmp, sizeof(tmp)) > 0) \
				snprintf((buf), USB_MAX_STRING, "%s", tmp); \
		}					\
	} while(0);
#  define USB_GET_IFACE_NAME(udev,iface,buf)		\
	do {						\
		if((iface)->desc.iInterface) {		\
			char	tmp[USB_MAX_STRING];	\
			if(usb_string((udev), (iface)->desc.iInterface, tmp, sizeof(tmp)) > 0) \
				snprintf((buf), USB_MAX_STRING, "%s", tmp); \
		}					\
	} while(0);
#endif

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/delay.h>	/* for udelay */
#include <linux/seq_file.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>
#include <asm/timex.h>
#include <linux/proc_fs.h>
#include <linux/usb.h>
#include "xpd.h"
#include "xproto.h"
#include "xbus-core.h"
#ifdef	DEBUG
#include "card_fxs.h"
#include "card_fxo.h"
#endif
#include "parport_debug.h"

static const char rcsid[] = "$Id: xpp_usb.c 3143 2007-10-16 19:17:46Z tzafrir $";

DEF_PARM(int, print_dbg, 0, 0644, "Print DBG statements");	/* must be before zap_debug.h */
DEF_PARM(int, usb1, 0, 0644, "Allow using USB 1.1 interfaces");
DEF_PARM_BOOL(rx_tasklet, 1, 0644, "Use receive tasklets");

#include "zap_debug.h"

/* FIXME: A flag that was deprecated at some point, and rather useless */
/* anyway. Only used in the code or-ed to other flags                  */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,14)
#  define URB_ASYNC_UNLINK 0
#endif
/* Get a minor range for your devices from the usb maintainer */
#define USB_SKEL_MINOR_BASE	192

#ifdef CONFIG_PROC_FS
#define	PROC_USBXPP_SUMMARY	"xpp_usb"
#endif

#ifdef	DEBUG_PCM_TIMING
static cycles_t	stamp_last_pcm_read;
static cycles_t accumulate_diff;
#endif

struct xusb_model_info;

struct xusb_endpoint {
	int		ep_addr;
	int		max_size;
	usb_complete_t	callback;
};

static int xusb_xframe_send(xbus_t *xbus, xframe_t *xframe);

static xbus_ops_t	xusb_ops = {
	.xframe_send = xusb_xframe_send,
	.xframe_new = NULL,			// Default allocator
	.xframe_free = NULL,			// Default deallocator
};

enum {
	XUSB_N_RX_FRAMES,
	XUSB_N_TX_FRAMES,
	XUSB_N_RX_ERRORS,
	XUSB_N_TX_ERRORS,
	XUSB_N_RCV_ZERO_LEN,
};

#define	XUSB_COUNTER(xusb, counter)	((xusb)->counters[XUSB_N_ ## counter])

#define	C_(x)	[ XUSB_N_ ## x ] = { #x }

static struct xusb_counters {
	char	*name;
} xusb_counters[] = {
	C_(RX_FRAMES),
	C_(TX_FRAMES),
	C_(RX_ERRORS),
	C_(TX_ERRORS),
	C_(RCV_ZERO_LEN),
};

#undef C_

#define	XUSB_COUNTER_MAX	ARRAY_SIZE(xusb_counters)

#define	MAX_PENDING_WRITES	100

enum xusb_dir {
	XUSB_RECV = 0,
	XUSB_SEND = 1,
};

/* Receive Tasklets */
#define	MAX_RECV_QUEUE	100

struct xframe_queue {
	struct list_head	head;
	unsigned int		count;
	unsigned int		worst_count;
	unsigned int		overflows;
	spinlock_t		lock;
};

static void xframe_queue_init(struct xframe_queue *q);
static int xframe_enqueue(struct xframe_queue *q, xframe_t *xframe);
static xframe_t *xframe_dequeue(struct xframe_queue *q);
static void receive_tasklet_func(unsigned long data);

/*
 * USB XPP Bus (a USB Device)
 */
typedef struct xpp_usb_bus {
	xbus_t			*xbus;
	struct usb_device	*udev;			/* save off the usb device pointer */
	struct usb_interface	*interface;		/* the interface for this device */
	unsigned char		minor;			/* the starting minor number for this device */

	struct xusb_model_info	*model_info;
	struct xusb_endpoint	endpoints[2];		/* RECV/SEND endpoints */

	struct urb		*read_urb;

	/* Receive tasklet */
	struct xframe_queue	receive_queue;
	struct tasklet_struct	receive_tasklet;
	int			cpu_rcv_intr[NR_CPUS];
	int			cpu_rcv_tasklet[NR_CPUS];

	struct completion	write_finished;		/* wait for the write to finish */

	int			present;		/* if the device is not disconnected */
	int			reading;		/* is the read_urb reading (listening) */
	atomic_t		pending_writes;		/* submited but not out yet */
	struct semaphore	sem;			/* locks this structure */
	int		counters[XUSB_COUNTER_MAX];

	/* metrics */
	struct timeval		last_tx;
	unsigned int		max_tx_delay;

#ifdef USB_FIELDS_MISSING
	/* storage for missing strings in old kernels */
	char			manufacturer[USB_MAX_STRING];
	char			product[USB_MAX_STRING];
	char			serial[USB_MAX_STRING];
	char			interface_name[USB_MAX_STRING];
#else
	const char		*manufacturer;
	const char		*product;
	const char		*serial;
	const char		*interface_name;
#endif

} xusb_t;

static	spinlock_t	xusb_lock = SPIN_LOCK_UNLOCKED;
static xusb_t *xusb_array[MAX_BUSES] = {};
static unsigned bus_count = 0;


/* prevent races between open() and disconnect() */
static DECLARE_MUTEX (disconnect_sem);

/*
 * Function Prototypes
 */
#if 0
static ssize_t xusb_read	(struct file *file, char *buffer, size_t count, loff_t *ppos);
static ssize_t xusb_write	(struct file *file, const char *buffer, size_t count, loff_t *ppos);
static int xusb_ioctl		(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int xusb_open		(struct inode *inode, struct file *file);
static int xusb_release		(struct inode *inode, struct file *file);
static void xusb_write_bulk_callback	(struct urb *urb, struct pt_regs *regs);
#endif

/*
 * AsteriskNow kernel has backported the "lean" callback from 2.6.20
 * to 2.6.19 without any macro to notify of this fact -- how lovely.
 * Debian-Etch and Centos5 are using 2.6.18 for now (lucky for us).
 * Fedora6 jumped from 2.6.18 to 2.6.20. So far luck is on our side ;-)
 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
#define	USB_PASS_CB(u)	struct urb *u, struct pt_regs *regs
#else
#define	USB_PASS_CB(u)	struct urb *u
#endif

static void xpp_urb_delete(struct urb *urb);
static struct urb *xpp_urb_new(xusb_t *dev, enum xusb_dir dir, size_t size);
static void xpp_send_callback(USB_PASS_CB(urb));
static void xpp_receive_callback(USB_PASS_CB(urb));

static int xusb_probe		(struct usb_interface *interface, const struct usb_device_id *id);
static void xusb_disconnect	(struct usb_interface *interface);
static int xusb_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data);

/*------------------------------------------------------------------*/

static int xusb_xframe_send(xbus_t *xbus, xframe_t *xframe)
{
	xusb_t			*xusb;
	struct urb		*urb;
	int			ret = 0;
	size_t			size;
	struct xusb_endpoint	*xusb_ep;

	BUG_ON(!xframe);
	BUG_ON(!xbus);
	xusb = xbus->priv;
	BUG_ON(!xusb);
	if(!xusb->present) {
		static	int rate_limit;

		if((rate_limit++ % 5000) == 0)
			XBUS_ERR(xbus, "USB device not present. Dropping packets (#%d).\n",
					rate_limit);
		ret = -ENODEV;
		goto freepack;
	}
	/*
	 * If something really bad happend, do not overflow the USB stack
	 */
	if(atomic_read(&xusb->pending_writes) > MAX_PENDING_WRITES) {
		static	int rate_limit;

		if((rate_limit++ % 5000) == 0)
			XBUS_ERR(xbus, "USB device is totaly stuck. Dropping packets (#%d).\n",
					rate_limit);
		ret = -ENODEV;
		goto freepack;
	}
	size = XFRAME_LEN(xframe);
	xusb_ep = &xusb->endpoints[XUSB_SEND];
	urb = xpp_urb_new(xusb, XUSB_SEND, size);
	if (!urb) {
		ERR("No free urbs available\n");
		ret = -ENOMEM;
		goto freepack;
	}

	/* FIXME: FIXME: FIXME: we use copy+free until low-level drivers allocate memory themselves */
	
	memcpy(urb->transfer_buffer, xframe->packets, size);

	ret = usb_submit_urb(urb, GFP_ATOMIC);
	if(ret < 0) {
		static int rate_limit;

		if((rate_limit++ % 1000) < 5)
			ERR("%s: failed submit_urb: %d\n", __FUNCTION__, ret);
		xpp_urb_delete(urb);
		ret = -EBADF;
		goto freepack;
	}
	if (print_dbg)
		dump_xframe("USB_FRAME_SEND", xbus, xframe);
	do_gettimeofday(&xusb->last_tx);
	atomic_inc(&xusb->pending_writes);
freepack:
	xbus->ops->xframe_free(xbus, xframe);	// FIXME: eventually will be done in the urb callback
	if(ret < 0)
		XUSB_COUNTER(xusb, TX_ERRORS)++;
	return ret;
}

static void xpp_urb_delete(struct urb *urb)
{
	BUG_ON(!urb);
	// DBG(GENERAL, "%s: (%d) %p %X", __FUNCTION__, urb->transfer_buffer_length, urb->transfer_buffer, urb->transfer_dma);
	usb_buffer_free (urb->dev, urb->transfer_buffer_length,
			urb->transfer_buffer,
			urb->transfer_dma);
	usb_free_urb(urb);
}

static struct urb *xpp_urb_new(xusb_t *xusb, enum xusb_dir dir, size_t size)
{
	struct usb_device	*udev = xusb->udev;
	struct xusb_endpoint	*xusb_ep = &xusb->endpoints[dir];
	unsigned int	ep_addr = xusb_ep->ep_addr;
	usb_complete_t	urb_cb = xusb_ep->callback;
	struct urb	*urb;
	unsigned char	*buffer;	/* the buffer to send data */
	unsigned int	epnum = ep_addr & USB_ENDPOINT_NUMBER_MASK;
	int		pipe = usb_pipein(ep_addr)
					? usb_rcvbulkpipe(udev, epnum)
					: usb_sndbulkpipe(udev, epnum);
		 
	if(size > xusb_ep->max_size)
		return NULL;
	urb = usb_alloc_urb(0, GFP_ATOMIC);
	if (!urb) {
		err("No free urbs available");
		return NULL;
	}

	/* on some platforms using this kind of buffer alloc
	 * call eliminates a dma "bounce buffer".
	 *
	 * NOTE: you'd normally want i/o buffers that hold
	 * more than one packet, so that i/o delays between
	 * packets don't hurt throughput. (Probably applies only to isochronous 
	 * transfers)
	 */
	urb->transfer_flags = (URB_NO_TRANSFER_DMA_MAP | URB_ASYNC_UNLINK);
	buffer = usb_buffer_alloc(udev, size, GFP_ATOMIC, &urb->transfer_dma);
	// DBG(GENERAL, "(%d) %p / %x", size, buffer, urb->transfer_dma);
	if (!buffer) {
		err("Couldn't allocate buffer");
		usb_free_urb(urb);
		return NULL;
	}
	usb_fill_bulk_urb(urb, udev, pipe, buffer, size, urb_cb, xusb);
	return urb;
}

/*------------------------- Receive Tasklet Handling ---------------*/

static void xframe_queue_init(struct xframe_queue *q)
{
	memset(q, 0, sizeof(*q));
	spin_lock_init(&q->lock);
	INIT_LIST_HEAD(&q->head);
}

static int xframe_enqueue(struct xframe_queue *q, xframe_t *xframe)
{
	unsigned long		flags;
	int			ret = 1;

	spin_lock_irqsave(&q->lock, flags);
	if(q->count >= MAX_RECV_QUEUE) {
		q->overflows++;
		ret = 0;
		goto out;
	}
	if(++q->count > q->worst_count)
		q->worst_count = q->count;
	list_add_tail(&xframe->frame_list, &q->head);
out:
	spin_unlock_irqrestore(&q->lock, flags);
	return ret;
}

static xframe_t *xframe_dequeue(struct xframe_queue *q)
{
	unsigned long		flags;
	xframe_t		*frm = NULL;
	struct list_head	*h;

	spin_lock_irqsave(&q->lock, flags);
	if(list_empty(&q->head))
		goto out;
	h = q->head.next;
	list_del(h);
	q->count--;
	frm = list_entry(h, xframe_t, frame_list);
out:
	spin_unlock_irqrestore(&q->lock, flags);
	return frm;
}

static void xusb_handle_xframe(xbus_t *xbus, xframe_t *xframe)
{
	xusb_t	*xusb;
	int	cpu = smp_processor_id();

	BUG_ON(!xbus);
	xusb = xbus->priv;
	BUG_ON(!xusb);
	xusb->cpu_rcv_intr[cpu]++;
	if(!xframe_enqueue(&xusb->receive_queue, xframe)) {
		xbus->ops->xframe_free(xbus, xframe);
		return;
	}
	tasklet_schedule(&xusb->receive_tasklet);
}

static void receive_tasklet_func(unsigned long data)
{
	xusb_t		*xusb = (xusb_t *)data;
	xbus_t		*xbus;
	xframe_t	*xframe = NULL;
	int		cpu = smp_processor_id();

	BUG_ON(!xusb);
	xbus = xusb->xbus;
	BUG_ON(!xbus);
	xusb->cpu_rcv_tasklet[cpu]++;
	while((xframe = xframe_dequeue(&xusb->receive_queue)) != NULL)
		xframe_receive(xbus, xframe);
}

static void init_receive_tasklet(xusb_t *xusb)
{
	xframe_queue_init(&xusb->receive_queue);
	tasklet_init(&xusb->receive_tasklet, receive_tasklet_func, (unsigned long)xusb);
}

/*------------------------- XPP USB Bus Handling -------------------*/

#define	XUSB_MODEL(interface, ep_in,ep_out,type,str)	\
	{					\
		.iface_num = (interface),	\
		.in = { .ep_addr = (ep_in) },	\
		.out = { .ep_addr = (ep_out) },	\
		.bus_type = (type),		\
		.desc = (str)			\
	}

static const struct xusb_model_info {
	const char		*desc;
	int			iface_num;
	struct xusb_endpoint	in;
	struct xusb_endpoint	out;
	xbus_type_t		bus_type;
} model_table[] = {
	XUSB_MODEL(0, 0x86, 0x02, FIRMWARE_XPP,		"FPGA_XPD"),
};

/* table of devices that work with this driver */
static const struct usb_device_id xusb_table [] = {
//	{ USB_DEVICE(0x04B4, 0x8613) }, // default of cypress
	{ USB_DEVICE(0xE4E4, 0x2211), .driver_info=(kernel_ulong_t)&model_table[0] }, // OLD FPGA
	{ USB_DEVICE(0xE4E4, 0x1132), .driver_info=(kernel_ulong_t)&model_table[0] }, // FPGA_FXS
	{ USB_DEVICE(0xE4E4, 0x1142), .driver_info=(kernel_ulong_t)&model_table[0] }, // FPGA_1141
	{ USB_DEVICE(0xE4E4, 0x1152), .driver_info=(kernel_ulong_t)&model_table[0] }, // FPGA_1151
	/* "Gadget Zero" firmware runs under Linux */
	//{ USB_DEVICE(0x0525, 0xa4a0) },
	{ }					/* Terminating entry */
};

MODULE_DEVICE_TABLE (usb, xusb_table);


/* usb specific object needed to register this driver with the usb subsystem */
static struct usb_driver xusb_driver = {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16)
	.owner =	THIS_MODULE,
#endif
	.name =		"xpp_usb",
	.probe =	xusb_probe,
	.disconnect =	xusb_disconnect,
	.id_table =	xusb_table,
};

/*
 * File operations needed when we register this driver.
 * This assumes that this driver NEEDS file operations,
 * of course, which means that the driver is expected
 * to have a node in the /dev directory. If the USB
 * device were for a network interface then the driver
 * would use "struct net_driver" instead, and a serial
 * device would use "struct tty_driver".
 */
static struct file_operations xusb_fops = {
	/*
	 * The owner field is part of the module-locking
	 * mechanism. The idea is that the kernel knows
	 * which module to increment the use-counter of
	 * BEFORE it calls the device's open() function.
	 * This also means that the kernel can decrement
	 * the use-counter again before calling release()
	 * or should the open() function fail.
	 */
	.owner =	THIS_MODULE,
};

/* 
 * usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with devfs and the driver core
 */
static struct usb_class_driver xusb_class = {
	.name =		"usb/xpp_usb%d",
	.fops =		&xusb_fops,
/* FIXME: The sysfs class interfase seems to have chaged around here */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
	.mode =		S_IFCHR | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH,
#endif
	.minor_base =	USB_SKEL_MINOR_BASE,
};

/*
 * Check that an endpoint's wMaxPacketSize attribute is 512. This
 * indicates that it is a USB2's high speed end point. 
 *
 * If it is 64, it means we have a USB1 controller. By default we do not
 * support it and just fail the probe of the device. However if the user
 * has set usb1=1, we continue and just put a notice.
 *
 * Returns true if all OK, false otherwise.
 */
static int check_usb1(struct usb_endpoint_descriptor *endpoint)
{
	const char *msg = (usb_pipein(endpoint->bEndpointAddress))?"input":"output";

	if(endpoint->wMaxPacketSize >= sizeof(xpacket_t))
		return 1;

	if(usb1) {
		NOTICE("USB1 endpoint detected: USB %s endpoint 0x%X support only wMaxPacketSize=%d.\n", 
				msg, endpoint->bEndpointAddress, endpoint->wMaxPacketSize);
		return 1;
	}
	NOTICE("USB1 endpoint detected. Device disabled. To enable: usb1=1, and read docs. (%s, endpoint %d, size %d).\n", 
			msg, endpoint->bEndpointAddress, endpoint->wMaxPacketSize);
	return 0;
}

/*
 * set up the endpoint information
 * check out the endpoints
 * FIXME: Should be simplified (above 2.6.10) to use usb_dev->ep_in[0..16] and usb_dev->ep_out[0..16]
 */
static int set_endpoints(xusb_t *xusb, struct usb_host_interface *iface_desc, struct xusb_model_info *model_info)
{
	struct usb_endpoint_descriptor	*endpoint;
	struct xusb_endpoint		*xusb_ep;
	int				ep_addr;
	int i;

#define	BULK_ENDPOINT(ep) (((ep)->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_BULK)

	for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) {
		endpoint = &iface_desc->endpoint[i].desc;
		ep_addr = endpoint->bEndpointAddress;
		
		if(!BULK_ENDPOINT(endpoint)) {
			DBG(GENERAL, "endpoint 0x%x is not bulk: mbAttributes=0x%X\n",
					ep_addr, endpoint->bmAttributes);
			continue;
		}
		if(usb_pipein(ep_addr)) {	// Input
			if(ep_addr == model_info->in.ep_addr) {
				if (!check_usb1(endpoint))
					return 0;
				xusb_ep = &xusb->endpoints[XUSB_RECV];
				xusb_ep->ep_addr = ep_addr;
				xusb_ep->max_size = endpoint->wMaxPacketSize;
				xusb_ep->callback = xpp_receive_callback;
			}
		} else {			// Output
			if(ep_addr == model_info->out.ep_addr) {
				if (!check_usb1(endpoint))
					return 0;
				xusb_ep = &xusb->endpoints[XUSB_SEND];
				xusb_ep->ep_addr = ep_addr;
				xusb_ep->max_size = endpoint->wMaxPacketSize;
				xusb_ep->callback = xpp_send_callback;
			}
		}
	}
	if (!xusb->endpoints[XUSB_RECV].ep_addr || !xusb->endpoints[XUSB_SEND].ep_addr) {
		ERR("Couldn't find bulk-in or bulk-out endpoints\n");
		return 0;
	}
	DBG(GENERAL, "in=0x%02X out=0x%02X\n", xusb->endpoints[XUSB_RECV].ep_addr, xusb->endpoints[XUSB_SEND].ep_addr);
	return 1;
}

/**
 *	xusb_probe
 *
 *	Called by the usb core when a new device is connected that it thinks
 *	this driver might be interested in.
 */
static int xusb_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	struct usb_device		*udev = interface_to_usbdev(interface);
	struct usb_host_interface	*iface_desc = usb_altnum_to_altsetting(interface, 0);
	xusb_t			*xusb = NULL;
	struct xusb_model_info	*model_info = (struct xusb_model_info*)id->driver_info;
	struct proc_dir_entry	*procsummary = NULL;
	xbus_t			*xbus = NULL;
	struct xusb_endpoint	*xusb_ep;
	unsigned long		flags;
	int			retval = -ENOMEM;
	size_t			xframe_size;
	int			i;
	
	DBG(GENERAL, "New XUSB device MODEL=%s bus_type=%d\n", model_info->desc, model_info->bus_type);
	if(iface_desc->desc.bInterfaceNumber != model_info->iface_num) {
		DBG(GENERAL, "Skip interface #%d != #%d\n",
			iface_desc->desc.bInterfaceNumber, model_info->iface_num);
		retval = -ENODEV;
		goto probe_failed;
	}

	/* The USB stack before 2.6.10 seems to be a bit shoddy. It seems that when being called
	 * from the probe we may already have the lock to udev (the Usb DEVice). Thus we call
	 * the internal __usb_reset_device instead. */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,10)
	if((retval = __usb_reset_device(udev)) < 0) {
#else
	if((retval =   usb_reset_device(udev)) < 0) {
#endif
		ERR("usb_reset_device failed: %d\n", retval);
		goto probe_failed;
	}
	if (!model_info) {
		ERR("Missing endpoint setup for this device %d:%d\n", 
				udev->descriptor.idVendor,udev->descriptor.idProduct);
		retval = -ENODEV;
		goto probe_failed;
	}

	/* allocate memory for our device state and initialize it */
	xusb = kmalloc(sizeof(xusb_t), GFP_KERNEL);
	if (xusb == NULL) {
		ERR("xpp_usb: Unable to allocate new xpp usb bus\n");
		retval = -ENOMEM;
		goto probe_failed;
	}
	memset(xusb, 0, sizeof(xusb_t));

	init_MUTEX (&xusb->sem);
	atomic_set(&xusb->pending_writes, 0);
	xusb->udev = udev;
	xusb->interface = interface;
	xusb->model_info = model_info;

	if(!set_endpoints(xusb, iface_desc, model_info)) {
		retval = -ENODEV;
		goto probe_failed;
	}
	xusb_ep = &xusb->endpoints[XUSB_RECV];

	/* Receive tasklets */
	init_receive_tasklet(xusb);

	xusb->read_urb = xpp_urb_new(xusb, XUSB_RECV, xusb_ep->max_size);
	if (!xusb->read_urb) {
		ERR("No free urbs available\n");
		retval = -ENOMEM;
		goto probe_failed;
	}
#ifndef USB_FIELDS_MISSING
	xusb->serial = udev->serial;
	xusb->manufacturer = udev->manufacturer;
	xusb->product = udev->product;
	xusb->interface_name = iface_desc->string;
#else
	USB_GET_STRING(udev, iSerialNumber, xusb->serial);
	USB_GET_STRING(udev, iManufacturer, xusb->manufacturer);
	USB_GET_STRING(udev, iProduct, xusb->product);
	USB_GET_IFACE_NAME(udev, iface_desc, xusb->interface_name);
#endif
	INFO("XUSB: manufacturer=[%s] product=[%s] serial=[%s] interface=[%s]\n",
		xusb->manufacturer, xusb->product, xusb->serial, xusb->interface_name);

	/* allow device read, write and ioctl */
	xusb->present = 1;

	/* we can register the device now, as it is ready */
	usb_set_intfdata (interface, xusb);
	retval = usb_register_dev (interface, &xusb_class);
	if (retval) {
		/* something prevented us from registering this driver */
		ERR ("Not able to get a minor for this device.\n");
		goto probe_failed;
	}

	xusb->minor = interface->minor;

	/* let the user know what node this device is now attached to */
	DBG(GENERAL, "USB XPP device now attached to minor %d\n", xusb->minor);

	/* Allocate high level structures */
	xframe_size = min(xusb->endpoints[XUSB_SEND].max_size , xusb->endpoints[XUSB_RECV].max_size);
	xbus = xbus_new(&xusb_ops, xframe_size);
	if(!xbus) {
		retval = -ENOMEM;
		goto probe_failed;
	}
	xbus->bus_type = model_info->bus_type;

	spin_lock_irqsave(&xusb_lock, flags);
	for(i = 0; i < MAX_BUSES; i++) {
		if(xusb_array[i] == NULL)
			break;
	}
	spin_unlock_irqrestore(&xusb_lock, flags);
	if(i >= MAX_BUSES) {
		ERR("xpp_usb: Too many XPP USB buses\n");
		retval = -ENOMEM;
		goto probe_failed;
	}
	{
		char	path[XBUS_DESCLEN];

		usb_make_path(udev, path, XBUS_DESCLEN);	// May trunacte... ignore
		snprintf(xbus->busdesc, XBUS_DESCLEN, "%s", path);
	}
	if(xusb->serial && xusb->serial[0])
		snprintf(xbus->serialnum, SERIALNUM_SIZE, "usb:%s", xusb->serial);
	DBG(GENERAL, "GOT XPP USB BUS #%d: %s (type=%d)\n", i, xbus->busdesc, xbus->bus_type);

	xusb_array[i] = xusb;


#ifdef CONFIG_PROC_FS
	DBG(PROC, "Creating proc entry " PROC_USBXPP_SUMMARY " in bus proc dir.\n");
	procsummary = create_proc_read_entry(PROC_USBXPP_SUMMARY, 0444, xbus->proc_xbus_dir,
			xusb_read_proc, xusb);
	if (!procsummary) {
		XBUS_ERR(xbus, "Failed to create proc file '%s'\n", PROC_USBXPP_SUMMARY);
		// FIXME: better error handling
		retval = -EIO;
		goto probe_failed;
	}
	procsummary->owner = THIS_MODULE;
#endif
	bus_count++;
	retval = usb_submit_urb(xusb->read_urb, GFP_ATOMIC);
	if(retval < 0) {
		ERR("%s: Failed to submit the receive URB errno=%d\n", __FUNCTION__, retval);
	}
	xusb->xbus = xbus;
	xbus->priv = xusb;
	xbus_activate(xbus);
	return retval;
probe_failed:
	ERR("Failed to initialize xpp usb bus: %d\n", retval);
	usb_set_intfdata (interface, NULL);
	if(xusb) {
		if(xusb->read_urb)
			xpp_urb_delete(xusb->read_urb);
		if(xusb->minor)	// passed registration phase
			usb_deregister_dev(interface, &xusb_class);
		kfree(xusb);
	}
	if(xbus) {
		if(procsummary) {
			XBUS_DBG(PROC, xbus, "Remove proc_entry: " PROC_USBXPP_SUMMARY "\n");
			remove_proc_entry(PROC_USBXPP_SUMMARY, xbus->proc_xbus_dir);
			procsummary = NULL;
		}
		xbus_disconnect(xbus);		// Blocking until fully deactivated!
	}
	return retval;
}

/**
 *	xusb_disconnect
 *
 *	Called by the usb core when the device is removed from the system.
 *
 *	This routine guarantees that the driver will not submit any more urbs
 *	by clearing dev->udev.  It is also supposed to terminate any currently
 *	active urbs.  Unfortunately, usb_bulk_msg(), used in xusb_read(), does
 *	not provide any way to do this.  But at least we can cancel an active
 *	write.
 */
static void xusb_disconnect(struct usb_interface *interface)
{
	xusb_t			*xusb;
	xbus_t			*xbus;
	int			minor;
	int			i;

	DBG(GENERAL, "CALLED\n");
	/* prevent races with open() */
	down (&disconnect_sem);

	xusb = usb_get_intfdata (interface);
	usb_set_intfdata (interface, NULL);
	xbus = xusb->xbus;

	/* find our xusb */
	for(i = 0; i < MAX_BUSES; i++) {
		if(xusb_array[i] == xusb)
			break;
	}
	BUG_ON(i >= MAX_BUSES);
	xusb_array[i] = NULL;

#ifdef CONFIG_PROC_FS
	if(xbus->proc_xbus_dir) {
		XBUS_DBG(GENERAL, xbus, "Remove proc_entry: " PROC_USBXPP_SUMMARY "\n");
		remove_proc_entry(PROC_USBXPP_SUMMARY, xbus->proc_xbus_dir);
	}
#endif
	tasklet_kill(&xusb->receive_tasklet);
	xusb->present = 0;
	xbus_disconnect(xbus);		// Blocking until fully deactivated!

	down (&xusb->sem);

	minor = xusb->minor;

	/* give back our minor */
	usb_deregister_dev (interface, &xusb_class);

	/* terminate an ongoing read */
	/* terminate an ongoing write */
	// FIXME: Does it really kill pending URB's?

	if(xusb->read_urb)
		xpp_urb_delete(xusb->read_urb);

	up (&xusb->sem);
	DBG(GENERAL, "Semaphore released\n");

	kfree(xusb);

	up (&disconnect_sem);
	DBG(GENERAL, "XUSB #%d now disconnected\n", minor);
}

static void xpp_send_callback(USB_PASS_CB(urb))
{
	xusb_t			*xusb = (xusb_t *)urb->context;
	xbus_t			*xbus = xusb->xbus;
	struct timeval		now;
	long			usec_diff;

	BUG_ON(!xbus);
	//flip_parport_bit(6);
	atomic_dec(&xusb->pending_writes);
	do_gettimeofday(&now);
	usec_diff =
		(now.tv_sec - xusb->last_tx.tv_sec)*1000*1000 +
		(now.tv_usec - xusb->last_tx.tv_usec);
	if(usec_diff > xusb->max_tx_delay)
		xusb->max_tx_delay = usec_diff;
	if(unlikely(usec_diff > 500)) {
		static int rate_limit;

		if((rate_limit++ % 5003) == 0)
			XBUS_NOTICE(xbus, "Slagish USB. %ld usec to transmit a frame\n",
				usec_diff);
	}
	/* sync/async unlink faults aren't errors */
	if (urb->status && !(urb->status == -ENOENT || urb->status == -ECONNRESET)) {
		static	int rate_limit;
		if((rate_limit++ % 1000) < 10)
			DBG(GENERAL, "nonzero read bulk status received: %d\n", urb->status);
		XUSB_COUNTER(xusb, TX_ERRORS)++;
	}
	xpp_urb_delete(urb);
	if(!xusb->present) {
		ERR("A packet from non-connected device?\n");
		return;
	}
	/* allow device read, write and ioctl */
	XUSB_COUNTER(xusb, TX_FRAMES)++;
}

static void xpp_receive_callback(USB_PASS_CB(urb))
{
	xusb_t		*xusb = (xusb_t *)urb->context;
	xbus_t		*xbus;
	xpacket_t	*pack;
	xframe_t	*xframe;
	size_t		size;
	int		retval;
	bool		do_resubmit = 1;
	bool		is_inuse = 0;
	struct timeval	now;

	do_gettimeofday(&now);
	BUG_ON(!xusb);
	xbus = xusb->xbus;
	if(!xbus) {
		NOTICE("spurious URB\n");
		return;
	}
	//flip_parport_bit(7);
	if (urb->status) {
		DBG(GENERAL, "nonzero read bulk status received: %d\n", urb->status);
		XUSB_COUNTER(xusb, RX_ERRORS)++;
		/* Free old URB, allocate a fresh one */
		if(xusb->read_urb)
			xpp_urb_delete(xusb->read_urb);
		xusb->read_urb = xpp_urb_new(xusb, XUSB_RECV, xusb->endpoints[XUSB_RECV].max_size);
		if (!xusb->read_urb) {
			ERR("URB allocation failed\n");
			do_resubmit = 0;;
		}
		goto end;
	}
	if(!down_read_trylock(&xbus->in_use)) {
		ERR("%s: xbus is going down\n", __FUNCTION__);
		do_resubmit = 0;
		goto end;
	}
	is_inuse = 1;
	if(!xusb->present) {
		ERR("A packet from non-connected device?\n");
		do_resubmit = 0;
		goto end;
	}
	size = urb->actual_length;
	if(size == 0) {
		static int	rate_limit;

		if((rate_limit++ % 5003) == 0)
			XBUS_NOTICE(xbus, "Received a zero length URBs (%d)\n", rate_limit);
		XUSB_COUNTER(xusb, RCV_ZERO_LEN)++;
		goto end;
	}
	xframe = xbus->ops->xframe_new(xbus, GFP_ATOMIC);
	if(!xframe) {
		ERR("%s: Not enough memory for packets. Dropping\n", __FUNCTION__);
		goto end;
	}
	atomic_set(&xframe->frame_len, size);
	pack = (xpacket_t *)xframe->packets;
	memcpy(xframe->packets, urb->transfer_buffer, size);
	xframe->tv_received = now;

	if (print_dbg)
		dump_xframe("USB_FRAME_RECEIVE", xbus, xframe);
	XUSB_COUNTER(xusb, RX_FRAMES)++;
	// Send UP
	if(rx_tasklet)
		xusb_handle_xframe(xbus, xframe);
	else
		xframe_receive(xbus, xframe);
end:
	if(is_inuse)
		up_read(&xbus->in_use);
	if(do_resubmit) {
		retval = usb_submit_urb(urb, GFP_ATOMIC);
		if (retval < 0) {
			ERR("failed re-submitting read urb, error %d\n", retval);
			return;
		}
	}
}


/*------------------------- Initialization -------------------------*/

int __init xpp_usb_init(void)
{
	int result;
	//xusb_t *xusb;

	INFO("revision %s\n", XPP_VERSION);

	/* register this driver with the USB subsystem */
	result = usb_register(&xusb_driver);
	if (result) {
		ERR("usb_register failed. Error number %d\n", result);
		return result;
	}
	return 0;
}


void __exit xpp_usb_cleanup(void)
{
	DBG(GENERAL, "\n");
	/* deregister this driver with the USB subsystem */
	usb_deregister(&xusb_driver);
}



#ifdef CONFIG_PROC_FS

static int xusb_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int 		len = 0;
	unsigned long	flags;
	int		i;
	//unsigned long stamp = jiffies;
	xusb_t		*xusb = data;

	if(!xusb)
		goto out;
	// TODO: probably needs a per-xusb lock:
	spin_lock_irqsave(&xusb_lock, flags);
	len += sprintf(page + len, "Device: %03d/%03d\n",
		xusb->udev->bus->busnum,
		xusb->udev->devnum
		);
	len += sprintf(page + len, "USB: manufacturer=%s\n", xusb->manufacturer);
	len += sprintf(page + len, "USB: product=%s\n", xusb->product);
	len += sprintf(page + len, "USB: serial=%s\n", xusb->serial);
	len += sprintf(page + len, "Minor: %d\n"
		"Model Info: Bus Type=%d (%s)\n",
		xusb->minor,
		xusb->model_info->bus_type,
		xusb->model_info->desc
		);
	len += sprintf(page + len, "Endpoints:\n"
		"\tIn:  0x%02X  - Size: %d)\n"
		"\tOut: 0x%02X  - Size: %d)\n",
		xusb->endpoints[XUSB_RECV].ep_addr,
		xusb->endpoints[XUSB_RECV].max_size,
		xusb->endpoints[XUSB_SEND].ep_addr,
		xusb->endpoints[XUSB_SEND].max_size
		);
	len += sprintf(page + len, "\npending_writes=%d\n", atomic_read(&xusb->pending_writes));
	len += sprintf(page + len, "max_tx_delay=%d\n", xusb->max_tx_delay);
	xusb->max_tx_delay = 0;
	if(rx_tasklet) {
		len += sprintf(page + len,
			"receive_queue: count = %5d, worst = %5d, overflows = %5d\n",
				xusb->receive_queue.count,
				xusb->receive_queue.worst_count,
				xusb->receive_queue.overflows);
		xusb->receive_queue.worst_count = 0;
		// xusb->receive_queue.overflows = 0;
	}
	len += sprintf(page + len, "\ncpu_rcv_intr:    ");
	for_each_online_cpu(i)
		len += sprintf(page + len, "%5d ", xusb->cpu_rcv_intr[i]);
	len += sprintf(page + len, "\ncpu_rcv_tasklet: ");
	for_each_online_cpu(i)
		len += sprintf(page + len, "%5d ", xusb->cpu_rcv_tasklet[i]);
	len += sprintf(page + len, "\n");
#ifdef	DEBUG_PCM_TIMING
	len += sprintf(page + len, "\nstamp_last_pcm_read=%lld accumulate_diff=%lld\n", stamp_last_pcm_read, accumulate_diff);
#endif
	len += sprintf(page + len, "\nCOUNTERS:\n");
	for(i = 0; i < XUSB_COUNTER_MAX; i++) {
		len += sprintf(page + len, "\t%-15s = %d\n", xusb_counters[i].name, xusb->counters[i]);
	}
#if 0
	len += sprintf(page + len, "<-- len=%d\n", len);
#endif
	spin_unlock_irqrestore(&xusb_lock, flags);
out:
	if (len <= off+count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;

}

#endif



MODULE_DESCRIPTION("XPP USB Driver");
MODULE_AUTHOR("Oron Peled <oron@actcom.co.il>");
MODULE_LICENSE("GPL");
MODULE_VERSION(XPP_VERSION);

module_init(xpp_usb_init);
module_exit(xpp_usb_cleanup);
