/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * simplesched.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002, 2003 Steve Underwood
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Id: simplesched.c,v 1.6 2005/09/03 15:10:18 steveu Exp $
 */

#include <stdio.h>
#include <inttypes.h>
#include <stdlib.h>
#include <sys/time.h>

#include "unicall.h"
#include "unicall/hashtable.h"
#include "unicall/unicall-private.h"

#define FALSE 0
#define TRUE (!FALSE)

int uc_schedule_event(uc_t *uc, int ms, void (*function)(uc_t *uc, void *data), void *data)
{
    int x;
    struct timeval tv;

    for (x = 0;  x < uc->uc_sched_max_to_date;  x++)
    {
        if (uc->uc_sched[x].callback == NULL)
            break;
        /*endif*/
    }
    /*endfor*/
    if (x >= uc->uc_sched_allocated)
    {
        uc->uc_sched_allocated += 5;
        uc->uc_sched = (uc_sched_t *) realloc (uc->uc_sched, sizeof(uc_sched_t)*uc->uc_sched_allocated);
    }
    /*endif*/
    if (x >= uc->uc_sched_max_to_date)
        uc->uc_sched_max_to_date = x + 1;
    /*endif*/
    gettimeofday(&tv, NULL);
    tv.tv_sec += ms/1000;
    tv.tv_usec += (ms%1000)*1000;
    if (tv.tv_usec > 1000000)
    {
        tv.tv_usec -= 1000000;
        tv.tv_sec += 1;
    }
    /*endif*/
    uc->uc_sched[x].when = tv;
    uc->uc_sched[x].callback = function;
    uc->uc_sched[x].data = data;
    return x;
}
/*- End of function --------------------------------------------------------*/

struct timeval *uc_schedule_next(uc_t *uc)
{
    struct timeval *closest = NULL;
    int x;

    for (x = 0;  x < uc->uc_sched_max_to_date;  x++)
    {
        if (uc->uc_sched[x].callback
            && 
                (!closest
                ||
                (closest->tv_sec > uc->uc_sched[x].when.tv_sec)
                ||
                    ((closest->tv_sec == uc->uc_sched[x].when.tv_sec)
                    && 
                    (closest->tv_usec > uc->uc_sched[x].when.tv_usec))))
        {
            closest = &uc->uc_sched[x].when;
        }
        /*endif*/
    }
    /*endfor*/
    return closest;
}
/*- End of function --------------------------------------------------------*/

void uc_schedule_run(uc_t *uc)
{
    struct timeval tv;
    int x;
    void (*callback)(uc_t *uc, void *data);
    void *data;

    gettimeofday(&tv, NULL);
    for (x = 0;  x < uc->uc_sched_max_to_date;  x++)
    {
        if (uc->uc_sched[x].callback
            &&
                ((uc->uc_sched[x].when.tv_sec < tv.tv_sec)
                ||
                    ((uc->uc_sched[x].when.tv_sec == tv.tv_sec)
                    &&
                    (uc->uc_sched[x].when.tv_usec <= tv.tv_usec))))
        {
            callback = uc->uc_sched[x].callback;
            data = uc->uc_sched[x].data;
            uc->uc_sched[x].callback = NULL;
            uc->uc_sched[x].data = NULL;
            callback(uc, data);
        }
        /*endif*/
    }
    /*endfor*/
}
/*- End of function --------------------------------------------------------*/

void uc_schedule_del(uc_t *uc, int id)
{
    if (id >= uc->uc_sched_max_to_date
        ||
        id < 0
        ||
        uc->uc_sched[id].callback == NULL)
    {
        uc_log(uc,
               UC_LOG_PROTOCOL_ERROR,
               "Received request to delete non-existant schedule id %d\n",
               id);
        return;
    }
    /*endif*/
    uc->uc_sched[id].callback = NULL;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
