//#define AUDIO_LOG
/*
 * libmfcr2 - a library for MFC/R2 signaling on E1 lines
 *
 * mfcr2.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002, 2003, 2004, 2005, 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mfcr2.c,v 1.55 2006/02/05 10:15:57 steveu Exp $
 */

/*! \file libmfcr2.h */

#define _ISOC9X_SOURCE  1
#define _ISOC99_SOURCE  1

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include <fcntl.h>
#include <stdlib.h>
#include <inttypes.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <zaptel/zaptel.h>

#include <tiffio.h>
#include <spandsp.h>

#include <unicall.h>
#include <unicall/hashtable.h>
#include <unicall/unicall-private.h>
#include <libsupertone.h>

#include "libmfcr2.h"
#include "mfcr2.h"

#define FALSE 0
#define TRUE (!FALSE)

/*
    Timers (specified in milliseconds):
    T1      The maximum time a forward tone can be on, from the outbound perspective. (15s)
    T2      The maximum time a forward tone can be off, from the outbound perspective. (15s)
    T3      The maximum time a whole compelled cycle can take, from the inbound perspective. (15s)
    
    T1A     This is not part of the R2 spec. It is the duration of non-compelled signalling
            pulses from the incoming side to the outgoing side. Some countries require such
            pulses, even though they break the compelled signalling scheme, and introduce timeout
            delays.
    T1B     This is not part of the R2 spec. It is a backstop timer used when the protocol
            variant calls for variable length digit strings to be handled by staying silent
            and letting the incoming end time out. Because the calling end is in control, any
            problems at the incoming end could cause the trunk to get stuck forever, unless this
            backstop timer is used to clear things up.
*/    

/* TODO: Still need options to:
- Use the direct answering method for incoming calls (caller's category will not be available)
- Generation and acceptance of billing pulses.
*/

#define DEFAULT_BLOCKING_RELEASE_TIME       450
#define DEFAULT_ANSWER_GUARD_TIME           100
#define DEFAULT_RELEASE_GUARD_TIME          20
#define DEFAULT_T1                          15000
#define DEFAULT_T1A                         150
#define DEFAULT_T1B                         60000
#define DEFAULT_T2                          24000
#define DEFAULT_T3                          15000

#define DEFAULT_MAX_SEIZE_ACK_WAIT          2000
#define DEFAULT_MAX_WAIT_FOR_GROUP_B_SIGNAL 15000
#define DEFAULT_MAX_AWAIT_ANSWER            60000

static uc_check_event_t check_event;
static uc_call_control_t call_control;
static uc_new_t create_new;
static uc_delete_t delete_context;
static uc_error_message_t error_message;
static uc_get_device_handles_t get_device_handles;
static uc_channel_open_t channel_open;
static uc_channel_close_t channel_close;
static uc_channel_set_api_codec_t channel_set_api_codec;
static uc_channel_write_t channel_write;
static uc_channel_flush_t channel_flush;
static uc_channel_gains_t channel_gains;
static uc_channel_echo_cancel_t channel_echo_cancel;
static uc_channel_switching_t channel_switching;

uc_protocol_t protocol_descriptor =
{
    "mfcr2",
    "E1 MFC/R2 signalling",
    "MFC/R2",
    "dz\0"
    "ar\0"
    "bh\0"
    "bo\0"
    "br\0"
    "cl\0"
    "cn\0"
    "co-land\0"
    "co-cell\0"
    "cz\0"
    "cd\0"
    "eg\0"
    "gh\0"
    "hn\0"
    "in\0"
    "id\0"
    "iq\0"
    "itu\0"
    "kr\0"
    "kw\0"
    "my\0"
    "mx\0"
    "ng\0"
    "pa\0"
    "ph\0"
    "ro\0"
    "sa\0"
    "sg\0"
    "th\0"
    "ve\0"
    "vn\0",
    "Algeria\0"
    "Argentina\0"
    "Bahrain\0"
    "Bolivia\0"
    "Brazil\0"
    "Chile\0"
    "China\0"
    "Colombia landlines\0"
    "Colombia cellular\0"
    "Czech\0"
    "Democratic Republic of Congo\0"
    "Egypt\0"
    "Ghana\0"
    "Honduras\0"
    "India\0"
    "Indonesia\0"
    "Iraq\0"
    "ITU\0"
    "Korea\0"
    "Kuwait\0"
    "Malaysia\0"
    "Mexico\0"
    "Nigeria\0"
    "Panama\0"
    "Philipinnes\0"
    "Romania\0"
    "Saudi Arabia\0"
    "Singapore\0"
    "Thailand\0"
    "Venezuela\0"
    "Vietnam\0",

    check_event,
    create_new,
    delete_context,
    call_control,
    error_message,
    get_device_handles,
    channel_open,
    channel_close,
    channel_set_api_codec,
    channel_write,
    channel_flush,
    channel_gains,
    channel_echo_cancel,
    channel_switching
};

/* 
There also appear to be R2 variants for at least the following:
    Australia
    Belgium
    Costa Rica
    Eastern Europe
    Ecuador (ITU)
    Ecuador (IME)
    Finland
    Greece
    Guatemala
    Israel
    New Zealand
    Paraguay
    Peru
    South Africa
    Uruguay
*/

enum
{
    ACTIVE_NONE,
    ACTIVE_SPEECH,
    ACTIVE_MF6,
    ACTIVE_SUPER_TONE
};

#define R2_STATUS_FORMAT "[%d/%-8s/%-14s/%-13s]"

static void mf_tone_on_event(uc_t *uc, int tone_id);
static void mf_tone_off_event(uc_t *uc);
static void abcd_signaling_event(uc_t *uc, int ch);

static int set_abcd_signal(uc_t *uc, int ch, uint8_t abcd);
static int set_mf_signal(uc_t *uc, int ch, int forward_signal);

static void select_active_rxtx(uc_t *uc, int which);
static void t1_expired(uc_t *uc, void *data);
static void t1a_expired(uc_t *uc, void *data);
static void t1b_expired(uc_t *uc, void *data);
static void t2_expired(uc_t *uc, void *data);
static void t3_expired(uc_t *uc, void *data);
static void seize_ack_wait_expired(uc_t *uc, void *data);
static void release_guard_expired(uc_t *uc, void *data);
static void far_unblocking_expired(uc_t *uc, void *data);
static void local_unblocking_expired(uc_t *uc, void *data);
static void answer_guard_expired(uc_t *uc, void *data);
static void clear_back_persistence_expired(uc_t *uc, void *data);
static void billing_pulse_expired(uc_t *uc, void *data);
static int send_calling_party_category_request(uc_t *uc);

static int abcd_control(uc_t *uc, int ch, int abcd);
static int check_abcd(uc_t *uc, int ch, int *abcd);
static int check_alarms(uc_t *uc, int ch, int *alarms);

static int variant_element(const char **variant, int which)
{
    int i;
    const char *s;
    const char *t;

    /* Find element "which" in a comma separated string of elements */
    t = *variant - 1;
    for (i = -1;  i < which;  i++)
    {
        s = t + 1;
        if ((t = strchr(t + 1, ',')) == NULL)
        {
            t = s + strlen(s);
            break;
        }
        /*endif*/
    }
    /*endfor*/
    if (i >= which - 1)
    {
        *variant = s;
        return  t - s;
    }
    /*endif*/
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int match_element(const char *s, int len, const char *t)
{
    if (strlen(t) == len  &&  strncmp(s, t, len) == 0)
        return 0;
    return -1;
}
/*- End of function --------------------------------------------------------*/

static int protocol_variant(const char *variant)
{
    int i;
    int inlen;
    int len;
    char const *s;
    char const *t;

    inlen = variant_element(&variant, 0);
    s = protocol_descriptor.variants;
    for (i = 1;  *s;  i++)
    {
        len = strlen(s);
        if (len == inlen  &&  strncmp(variant, s, len) == 0)
            return  i;
        /*endif*/
        s += (len + 1);
    }
    /*endfor*/
    return  -1;
}
/*- End of function --------------------------------------------------------*/

static char *r2_state_to_str(mfcr2_signaling_state_t *mfcr2)
{
    switch (mfcr2->r2_state)
    {
    case 0:
        return "Idle";
    case R2_BACK_SEIZE_ACK:
    case R2_FWD_SEIZE_ACK:
        switch (mfcr2->mfc_group)
        {
        case MFC_BACK_STARTING:
        case MFC_FWD_STARTING:
            return "Seize ack";
        case MFC_BACK_GROUP_A:
            return "Group A";
        case MFC_BACK_GROUP_B:
            return "Group B";
        case MFC_BACK_GROUP_C:
            return "Group C";
        case MFC_FWD_GROUP_I:
            return "Group I";
        case MFC_FWD_GROUP_II:
            return "Group II";
        case MFC_FWD_GROUP_III:
            return "Group III";
        }
        /*endswitch*/
        break;
    case R2_BACK_ANSWER:
        return "Answer";
    case R2_BACK_CLEAR_BACK:
        return "Clear back";
    case R2_BACK_CLEAR_FWD:
        return "Clear fwd";
    case R2_BACK_RELEASE_GUARD:
        return "Release guard";
    case R2_FWD_SEIZE:
        return "Seize";
    case R2_FWD_AWAIT_ANSWER:
        return "Await answer";
    case R2_FWD_ANSWERED_BEFORE_MFC_COMPLETE:
        return "Answered pre B";
    case R2_FWD_ANSWERED:
        return "Answered";
    case R2_FWD_CLEAR_BACK_BEFORE_ANSWER:
        return "Clear back A";
    case R2_FWD_CLEAR_BACK_AFTER_ANSWER:
        return "Clear back B";
    case R2_FWD_CLEAR_FWD_BEFORE_SEIZE_ACK:
        return "Clear fwd A";
    case R2_FWD_CLEAR_FWD_BEFORE_ANSWER:
        return "Clear fwd B";
    case R2_FWD_CLEAR_FWD_AFTER_ANSWER:
        return "Clear fwd C";
    case R2_FWD_CLEAR_FWD_AFTER_CLEAR_BACK:
        return "Clear fwd D";
    }
    /*endswitch*/
    return "???";
}
/*- End of function --------------------------------------------------------*/

static char *mfc_state_to_str(int state)
{
    switch (state)
    {
    case 0:
        return "Idle";
    case MFC_SENT_SEIZE_ACK:
        return "Seize ack";
    case MFC_SENT_CATEGORY_REQUEST:
        return "Category req";
    case MFC_SENT_DNIS_REQUEST:
        return "DNIS request";
    case MFC_SENT_ANI_REQUEST:
        return "ANI request";
    case MFC_SENT_CHANGE_TO_GROUP_II:
        return "Go to grp II";
    case MFC_SENT_ACCEPTED_CHARGE:
        return "Accepted Paid";
    case MFC_SENT_ACCEPTED_NO_CHARGE:
        return "Accepted Free";
    case MFC_SENT_NETWORK_CONGESTION:
        return "Congestion";
    case MFC_SENT_UNASSIGNED_NUMBER:
        return "Unassigned";
    case MFC_SENT_DEST_BUSY:
        return "Busy";
    case MFC_SENT_DEST_OUT_OF_ORDER:
        return "Out of order";
    case MFC_SENT_CATEGORY:
        return "Category";
    case MFC_SENT_DNIS:
        return "DNIS";
    case MFC_SENT_ANI:
        return "ANI";
    case MFC_SENT_END_OF_DNIS:
        return "End of DNIS";
    case MFC_SENT_END_OF_ANI:
        return "End of ANI";
    case MFC_SENT_NOTHING:
        return "Silent";
    }
    /*endswitch*/
    return "???";
}
/*- End of function --------------------------------------------------------*/

static void clear_outgoing_mf_cycle_timeouts(uc_t *uc)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (mfcr2->t1_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->t1_timer);
        mfcr2->t1_timer = -1;
    }
    /*endif*/
    if (mfcr2->t2_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->t2_timer);
        mfcr2->t2_timer = -1;
    }
    /*endif*/
    if (mfcr2->t1b_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->t1b_timer);
        mfcr2->t1b_timer = -1;
    }
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

static void select_active_rxtx(uc_t *uc, int which)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    /* The rx and tx active selections are ganged */
    uc->chan[ch].active_rx =
    uc->chan[ch].active_tx = which;
    if (which != ACTIVE_MF6)
    {
        /* Any tone generation and detection should stop at any change */
        mfcr2->mf_rx_signal = 0;
        set_mf_signal(uc, ch, 0);
        /* Any outstanding tone cycle timers should be zapped at this point */
        clear_outgoing_mf_cycle_timeouts(uc);
        if (mfcr2->t3_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->t3_timer);
            mfcr2->t3_timer = -1;
        }
        /*endif*/
    }
    /*endif*/
    mfcr2->mf_rx_signal = 0;
}
/*- End of function --------------------------------------------------------*/

static void protocol_error_recovery(uc_t *uc, int ch, int cause)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc,
           UC_LOG_PROTOCOL_ERROR,
           "R2 prot. err. " R2_STATUS_FORMAT " cause %d - %s\n",
           uc->chan[ch].active_rx,
           uc_state2str(uc->chan[ch].state),
           r2_state_to_str(mfcr2),
           mfc_state_to_str(mfcr2->mfc_state),
           cause,
           error_message(uc, cause));
    /* If we are blocked, a protocol problem isn't a good reason to unblock.
       However, for other starting conditions, going to the idle state seems OK. */
    if (uc->chan[ch].state != UC_STATE_BLOCKED)
        uc->chan[ch].state = UC_STATE_IDLE;
    /*endif*/

    /* Settle everything back to its initial state */
    mfcr2->r2_state = R2_IDLE;
    mfcr2->mfc_group = 0;
    mfcr2->mfc_state = 0;
    mfcr2->calling_party_category_exchanged = FALSE;
    select_active_rxtx(uc, ACTIVE_SPEECH);
    set_abcd_signal(uc, ch, mfcr2->abcd_idle);
    set_mf_signal(uc, ch, 0);

    if (mfcr2->far_unblocking_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->far_unblocking_timer);
        mfcr2->far_unblocking_timer = -1;
    }
    /*endif*/
    if (mfcr2->local_unblocking_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->local_unblocking_timer);
        mfcr2->local_unblocking_timer = -1;
    }
    /*endif*/
    if (mfcr2->seize_ack_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->seize_ack_timer);
        mfcr2->seize_ack_timer = -1;
    }
    /*endif*/
    if (mfcr2->answer_guard_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->answer_guard_timer);
        mfcr2->answer_guard_timer = -1;
    }
    /*endif*/
    if (mfcr2->await_answer_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->await_answer_timer);
        mfcr2->await_answer_timer = -1;
    }
    /*endif*/
    if (mfcr2->wait_for_group_b_signal_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->wait_for_group_b_signal_timer);
        mfcr2->wait_for_group_b_signal_timer = -1;
    }
    /*endif*/
    if (mfcr2->release_guard_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->release_guard_timer);
        mfcr2->release_guard_timer = -1;
    }
    /*endif*/
    if (mfcr2->clear_back_persistence_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->clear_back_persistence_timer);
        mfcr2->clear_back_persistence_timer = -1;
    }
    /*endif*/
    if (mfcr2->billing_pulse_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->billing_pulse_timer);
        mfcr2->billing_pulse_timer = -1;
    }
    /*endif*/
    clear_outgoing_mf_cycle_timeouts(uc);
    if (mfcr2->t3_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->t3_timer);
        mfcr2->t3_timer = -1;
    }
    /*endif*/

    ev.e = UC_EVENT_PROTOCOLFAIL;
    ev.gen.channel = ch;
    ev.gen.data = cause;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

int set_mf_signal(uc_t *uc, int ch, int signal)
{
    mfcr2_signaling_state_t *mfcr2;
    int x;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (signal == 0  &&  mfcr2->mf_tx_signal)
    {
        /* Flush any tone in the transmit buffer, to speed up the next step. */
        x = ZT_FLUSH_WRITE;
        if (ioctl(uc->chan[ch].handle, ZT_FLUSH, &x))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  -1;
        }
        /*endif*/
    }
    /*endif*/
    if (mfcr2->call  &&  !(mfcr2->call->incoming_call))
    {
        clear_outgoing_mf_cycle_timeouts(uc);
        if (signal)
            mfcr2->t1_timer = uc_schedule_event(uc, mfcr2->t1, t1_expired, (void *) (intptr_t) ch);
        else
            mfcr2->t2_timer = uc_schedule_event(uc, mfcr2->t2, t2_expired, (void *) (intptr_t) ch);
        /*endif*/
    }
    /*endif*/
    if ((mfcr2->mf_tx_signal & 0x7F) != signal)
    {
        uc_log(uc,
               UC_LOG_DEBUG_2,
               "%c %s ->      " R2_STATUS_FORMAT "\n",
               (signal)  ?  signal   :  (mfcr2->mf_tx_signal & 0x7F),
               (signal)  ?  "on "  :  "off",
               uc->chan[ch].active_rx,
               uc_state2str(uc->chan[ch].state),
               r2_state_to_str(mfcr2),
               mfc_state_to_str(mfcr2->mfc_state));
        r2_mf_tx_put(&mfcr2->mf_tx_state, signal);
        mfcr2->mf_tx_signal = signal;
    }
    /*endif*/
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int set_abcd_signal(uc_t *uc, int ch, uint8_t abcd)
{
    mfcr2_signaling_state_t *mfcr2;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    abcd |= mfcr2->base_abcd_bits;
    uc_log(uc,
           UC_LOG_DEBUG_2,
           "%d%d%d%d  ->      " R2_STATUS_FORMAT "\n",
           (abcd >> 3) & 1, (abcd >> 2) & 1, (abcd >> 1) & 1, abcd & 1,
           uc->chan[ch].active_rx,
           uc_state2str(uc->chan[ch].state),
           r2_state_to_str(mfcr2),
           mfc_state_to_str(mfcr2->mfc_state));
    if (abcd_control(uc, ch, abcd) != UC_RET_OK)
    {
        protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_SET_CAS_FAILURE);
        return  -1;
    }
    /*endif*/
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int sanity_check(uc_t *uc, uc_call_t *call)
{
    /* A temporary fudge, until the root causes of some anomalies are fixed */
    if (uc == NULL  ||  call == NULL  ||  (call->chan >= uc->num_channels))
        return 0;
    /*endif*/
    return 1;
}
/*- End of function --------------------------------------------------------*/

static void start_detected(uc_t *uc, int ch)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    uc_call_t *call;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Detected\n");

    /* TODO: a memory allocation issue at this point is not handled well. */
    if ((call = uc_createcall(uc, 0)) == NULL)
    {
        uc_log(uc, UC_LOG_ERROR, "Failed to allocate a new call structure\n");
        return;
    }
    /*endif*/
    mfcr2->call = call;

    call->incoming_call = TRUE;

    /* Right now we have no received call information. */
    mfcr2->ani_cnt = 0;
    mfcr2->ani_len = mfcr2->ani_max_rx_digits;
    mfcr2->ani_complete = FALSE;
    mfcr2->dnis_cnt = 0;
    mfcr2->dnis_len = mfcr2->dnis_max_rx_digits;
    mfcr2->dnis_complete = FALSE;
    mfcr2->calling_party_category_exchanged = FALSE;

    call->chan = ch;
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_DETECTED;
    /*endif*/
    call->state = UC_STATE_DETECTED;
    /* Set a default for the final disconnection */
    call->disconnect_reason = UC_CAUSE_UNSPECIFIED_CAUSE;
    /* We need to choose the codec used by the card. */
    call->callparms.userinfo_layer1_protocol = UC_LAYER_1_ALAW;

    r2_mf_tx_init(&mfcr2->mf_tx_state, FALSE);
    r2_mf_rx_init(&mfcr2->mf_rx_state, TRUE);
    select_active_rxtx(uc, ACTIVE_MF6);
    /* The seize ack tells the other end our MF6 detector is
       active, and tone signaling may begin. */
    mfcr2->r2_state = R2_BACK_SEIZE_ACK;
    mfcr2->mfc_group = MFC_BACK_STARTING;
    mfcr2->mfc_state = MFC_SENT_SEIZE_ACK;
    if (set_abcd_signal(uc, ch, mfcr2->back_abcd_seize_ack))
        return;
    /*endif*/

    ev.e = UC_EVENT_DETECTED;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_call_disconnected(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc,
           UC_LOG_FLOW,
           "Call disconnected(cause=%s [%d]) - state 0x%X\n",
           uc_cause2str(call->disconnect_reason),
           call->disconnect_reason,
           call->state);
    
    /* Make sure any tone processing is stopped, if we disconnected during
       call setup. */
    select_active_rxtx(uc, ACTIVE_SPEECH);

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_CALL_DISCONNECTED;
    /*endif*/
    call->state = UC_STATE_CALL_DISCONNECTED;

    ev.fardisconnected.e = UC_EVENT_DROPCALL;
    ev.fardisconnected.channel = call->chan;
    ev.fardisconnected.cause = call->disconnect_reason;
    ev.fardisconnected.crn = call->crn;
    ev.fardisconnected.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void start_far_disconnected(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    if (!sanity_check(uc, call))
    {
        uc_log(uc, UC_LOG_ERROR, "!!!! start_far_disconnected for an invalid call!\n");
        return;
    }
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc,
           UC_LOG_FLOW,
           "Far end disconnected(cause=%s [%d]) - state 0x%X\n",
           uc_cause2str(call->disconnect_reason),
           call->disconnect_reason,
           call->state);

    /* Don't stop the tone detector here. */

    if (mfcr2->wait_for_group_b_signal_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->wait_for_group_b_signal_timer);
        mfcr2->wait_for_group_b_signal_timer = -1;
    }
    /*endif*/
    if (mfcr2->seize_ack_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->seize_ack_timer);
        mfcr2->seize_ack_timer = -1;
    }
    /*endif*/
    if (mfcr2->await_answer_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->await_answer_timer);
        mfcr2->await_answer_timer = -1;
    }
    /*endif*/
    if (mfcr2->answer_guard_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->answer_guard_timer);
        mfcr2->answer_guard_timer = -1;
    }
    /*endif*/
    if (mfcr2->clear_back_persistence_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->clear_back_persistence_timer);
        mfcr2->clear_back_persistence_timer = -1;
    }
    /*endif*/
    if (mfcr2->billing_pulse_timer >= 0)
    {
        uc_schedule_del(uc, mfcr2->billing_pulse_timer);
        mfcr2->billing_pulse_timer = -1;
    }
    /*endif*/
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_FAR_DISCONNECTED;
    /*endif*/
    call->state = UC_STATE_FAR_DISCONNECTED;

    ev.fardisconnected.e = UC_EVENT_FARDISCONNECTED;
    ev.fardisconnected.channel = call->chan;
    ev.fardisconnected.cause = call->disconnect_reason;
    ev.fardisconnected.crn = call->crn;
    ev.fardisconnected.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void start_idle(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    /* N.B. This is really only for starting the idle state from a call. From
       initialisation, unblocking, or error recovery this is not the right
       choice. */
    /* Make sure no junk was left behind. */
    mfcr2->r2_state = R2_IDLE;
    mfcr2->mfc_group = 0;
    mfcr2->mfc_state = 0;
    mfcr2->calling_party_category_exchanged = FALSE;
    select_active_rxtx(uc, ACTIVE_SPEECH);

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_IDLE;
    /*endif*/
    call->state = UC_STATE_IDLE;

    ev.e = UC_EVENT_RELEASECALL;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));

    /* Although we destroy the call here, the event sent just above may not
       have been delivered. It is important, therefore, that the application
       only treat the call as a reference tag, and does not try to look at
       its contents. */
    uc_destroycall(uc, call->crn);
    mfcr2->call = NULL;
}
/*- End of function --------------------------------------------------------*/

static void report_dialed_number(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    uc_log(uc, UC_LOG_FLOW, "DNIS now '%s'\n", call->callparms.destination_number);
    memset (&ev.offered, '\0', sizeof(ev.offered));
    ev.e = UC_EVENT_DIALEDNUMBER;
    ev.offered.channel = call->chan;
    strcpy(ev.offered.parms.originating_number, call->callparms.originating_number);
    strcpy(ev.offered.parms.destination_number, call->callparms.destination_number);
    ev.offered.parms.calling_party_category = call->callparms.calling_party_category;
    ev.offered.parms.call_type = call->callparms.call_type;
    ev.offered.parms.userinfo_layer1_protocol = call->callparms.userinfo_layer1_protocol;
    ev.offered.crn = call->crn;
    ev.offered.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void start_offered(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_OFFERED;
    /*endif*/
    call->state = UC_STATE_OFFERED;
    
    call->disconnect_reason = UC_CAUSE_NORMAL_CLEARING;

    memset (&ev.offered, '\0', sizeof(ev.offered));
    ev.offered.e = UC_EVENT_OFFERED;
    ev.offered.channel = call->chan;
    strcpy (ev.offered.parms.originating_number, call->callparms.originating_number);
    strcpy (ev.offered.parms.destination_number, call->callparms.destination_number);
    ev.offered.parms.calling_party_category = call->callparms.calling_party_category;
    ev.offered.parms.call_type = call->callparms.call_type;
    ev.offered.parms.userinfo_layer1_protocol = call->callparms.userinfo_layer1_protocol;
    ev.offered.crn = call->crn;
    ev.offered.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void inquire_for_more_info(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    ev.e = UC_EVENT_REQUESTMOREINFO;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_proceeding(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_PROCEEDING;
    /*endif*/
    call->state = UC_STATE_PROCEEDING;

    ev.e = UC_EVENT_PROCEEDING;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_alerting(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    select_active_rxtx(uc, ACTIVE_SPEECH);
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_ALERTING;
    /*endif*/
    call->state = UC_STATE_ALERTING;

    ev.e = UC_EVENT_ALERTING;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_connected(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_CONNECTED;
    /*endif*/
    call->state = UC_STATE_CONNECTED;
    
    select_active_rxtx(uc, ACTIVE_SPEECH);
    /* Set the default for the clear that will occur later */
    call->disconnect_reason = UC_CAUSE_NORMAL_CLEARING;

    if (call->incoming_call)
    {
        ev.e = UC_EVENT_ANSWERED;
    }
    else
    {
        if (mfcr2->await_answer_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->await_answer_timer);
            mfcr2->await_answer_timer = -1;
        }
        /*endif*/
        ev.e = UC_EVENT_CONNECTED;
    }
    /*endif*/
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static int start_clear_fwd(uc_t *uc)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Clearing fwd\n");

    mfcr2->mfc_state = 0;
    if (set_abcd_signal(uc, ch, mfcr2->fwd_abcd_clear_fwd))
        return  UC_RET_BAD_DEVICE;
    /*endif*/
    mfcr2->calling_party_category_exchanged = FALSE;
    select_active_rxtx(uc, ACTIVE_SPEECH);
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static void seize_ack_wait_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "seize_ack_wait_expired\n");
    mfcr2->seize_ack_timer = -1;
    protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_SEIZE_ACK_TIMEOUT);
}
/*- End of function --------------------------------------------------------*/

static void release_guard_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_call_t *call;

    call = (uc_call_t *) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Release guard expired\n");
    mfcr2->release_guard_timer = -1;
    start_idle(uc, call);
}
/*- End of function --------------------------------------------------------*/

static void answer_guard_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    uc_call_t *call;
    int ch;

    call = (uc_call_t *) data;
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Answer guard expired\n");
    /* The answer guard time ensures we do not answer with the CAS bits before
       the audio buffer has flushed tone out to the far end. Some MFC/R2
       kit doesn't like this kind of out of sequence answering. */
    /* TODO: Waiting for the tone to drain from the write buffer by the buffer
             empty event would be crisper. Adding some margin to the end of that
             might make things more tolerant, though. */
    mfcr2->answer_guard_timer = -1;
    /* If answer occured before the timeout, we apply the answer condition now. */
    if (call->state == UC_STATE_CONNECTED)
    {
        mfcr2->r2_state = R2_BACK_ANSWER;
        if (set_abcd_signal(uc, ch, mfcr2->back_abcd_answer))
            return;
        /*endif*/
        start_connected(uc, mfcr2->call);
    }
    else
    {
        if (call->chan >= 0)
            uc->chan[call->chan].state = UC_STATE_ACCEPTED;
        /*endif*/
        call->state = UC_STATE_ACCEPTED;

        ev.e = UC_EVENT_ACCEPTED;
        ev.gen.channel = ch;
        ev.gen.crn = call->crn;
        ev.gen.call = call;
        uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
    }
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

static void clear_back_persistence_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_call_t *call;
    int ch;
    
    call = (uc_call_t *) data;
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    mfcr2->clear_back_persistence_timer = -1;
    start_far_disconnected(uc, mfcr2->call);
}
/*- End of function --------------------------------------------------------*/

static void billing_pulse_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_call_t *call;
    int ch;
    
    call = (uc_call_t *) data;
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    mfcr2->billing_pulse_timer = -1;
    /* Return to the answer state, if we are still connected. */
    if (call->state == UC_STATE_CONNECTED)
        set_abcd_signal(uc, ch, mfcr2->back_abcd_answer);
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

static void t1_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    mfcr2->t1_timer = -1;
    /* We should only get here if the tone playing timed out. This would be
       a protocol error, since the compelled signaling process should not
       allow a tone to play for so long. */
    protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_T1_TIMEOUT);
}
/*- End of function --------------------------------------------------------*/

static void t1a_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (set_mf_signal(uc, ch, 0))
        return;
    /*endif*/
    mfcr2->t1a_timer = -1;
}
/*- End of function --------------------------------------------------------*/

static void t1b_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    mfcr2->t1b_timer = -1;
    /* We should only get here if the far end failed to time out and move on. */
    protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_T1B_TIMEOUT);
}
/*- End of function --------------------------------------------------------*/

static void t2_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    mfcr2->t2_timer = -1;
    /* We should only get here if there was too long a gap between forward
       tones. This would be a protocol error, since the compelled signaling
       process should not go quiet for so long. */
    protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_T2_TIMEOUT);
}
/*- End of function --------------------------------------------------------*/

static void t3_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    mfcr2->t3_timer = -1;
    if (mfcr2->group_b_on_DNIS_timeout
        &&
        mfcr2->mfc_group == MFC_BACK_GROUP_A
        &&
        mfcr2->mfc_state == MFC_SENT_DNIS_REQUEST)
    {
        /* The outgoing end has run out of DNIS digits to send us. Proceed
           to plan B..... er, group B. */
        mfcr2->dnis_complete = TRUE;
        if (mfcr2->get_ani_after_dnis)
        {
            send_calling_party_category_request(uc);
        }
        else
        {
            mfcr2->mfc_group = MFC_BACK_GROUP_B;
            mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
            if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                return;
            /*endif*/
        }
        /*endif*/
        if (mfcr2->t1a_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->t1a_timer);
            mfcr2->t1a_timer = -1;
        }
        /*endif*/
        mfcr2->t1a_timer = uc_schedule_event(uc, mfcr2->t1a, t1a_expired, (void *) (intptr_t) ch);
        return;
    }
    /*endif*/
    /* We should only get here if a compelled cycle took too long. This would
       be a protocol error, since the compelled signaling process should never
       take so long. */
    protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_T3_TIMEOUT);
}
/*- End of function --------------------------------------------------------*/

static void far_unblocking_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    int ch;
    
    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "far_unblocking_expired\n");

    mfcr2->far_unblocking_timer = -1;

    mfcr2->far_blocked = FALSE;
    if (uc->chan[ch].state == UC_STATE_BLOCKED)
        uc->chan[ch].state = UC_STATE_IDLE;
    /*endif*/

    ev.e = UC_EVENT_FARUNBLOCKED;
    ev.gen.channel = ch;
    ev.gen.crn = 0;
    ev.gen.call = NULL;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void local_unblocking_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    int ch;
    
    ch = (intptr_t) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "local_unblocking_expired\n");

    mfcr2->local_unblocking_timer = -1;

    mfcr2->local_blocked = FALSE;

    ev.e = UC_EVENT_LOCALUNBLOCKED;
    ev.gen.channel = ch;
    ev.gen.crn = 0;
    ev.gen.call = NULL;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void wait_for_group_b_signal_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_call_t *call;
    
    call = (uc_call_t *) data;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Timed out waiting for group B signal\n");

    mfcr2->wait_for_group_b_signal_timer = -1;

    /* Treat this as though the other end cleared the call. */
    call->disconnect_reason = UC_CAUSE_TEMPORARY_FAILURE;
    start_far_disconnected(uc, call);
}
/*- End of function --------------------------------------------------------*/

static void await_answer_expired(uc_t *uc, void *data)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_call_t *call;
    
    call = (uc_call_t *) data;    
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Timed out waiting for answer\n");

    mfcr2->await_answer_timer = -1;

    if (!sanity_check(uc, call))
    {
        uc_log(uc, UC_LOG_ERROR, "!!!! await_answer_expired for an invalid call!\n");
        return;
    }
    /*endif*/
    /* Treat this as though the other end cleared the call. */
    call->disconnect_reason = UC_CAUSE_NO_ANSWER_FROM_USER;
    start_far_disconnected(uc, call);
}
/*- End of function --------------------------------------------------------*/

static int send_ani_digit(uc_t *uc, int ch, int which)
{
    mfcr2_signaling_state_t *mfcr2;
    int forwardsig;
    
    /* Which = 0 for send next digit (n)
             = 1 for send n-1
             = 2 for send n-2
             = 3 for send n-3
             = 4 for start from the beginning
    */
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (which)
    {
        /* We need to hop back and resend some digits, rather than just send
           the next digit */
        if (which <= 3  &&  mfcr2->ani_cnt > which)
        {
            /* Resend n-1, n-2 or n-3 */
            mfcr2->ani_cnt -= (which + 1);
        }
        else
        {
            /* Restart from the beginning */
            mfcr2->ani_cnt = 0;
        }
        /*endif*/
    }
    /*endif*/
    if (mfcr2->ani_cnt < mfcr2->ani_len)
    {
        mfcr2->mfc_state = MFC_SENT_ANI;
        forwardsig = mfcr2->call->callparms.originating_number[mfcr2->ani_cnt++];
        if (set_mf_signal(uc, ch, forwardsig))
            return  -1;
        /*endif*/
    }
    else
    {
        if (mfcr2->group_i_end_of_ANI
            ||
            (mfcr2->ani_len == 0  &&  mfcr2->group_i_end_of_ANI_restricted))
        {
            mfcr2->mfc_state = MFC_SENT_END_OF_ANI;
            if (mfcr2->ani_len == 0  &&  mfcr2->group_i_end_of_ANI_restricted)
            {
                if (set_mf_signal(uc, ch, mfcr2->group_i_end_of_ANI_restricted))
                    return  -1;
                /*endif*/
            }
            else
            {
                if (set_mf_signal(uc, ch, mfcr2->group_i_end_of_ANI))
                    return  -1;
                /*endif*/
            }
        }
        else
        {
            /* Send nothing, and let the timeout which will occur at the far
               end cause things to continue. */
            clear_outgoing_mf_cycle_timeouts(uc);
            /* We started this call, so we are in control. If the far end doesn't time
               out properly, it may never respond, and we will be stuck forever.
               Use a backstop timer, so things will recover if the far end gets stuck. */
            mfcr2->t1b_timer = uc_schedule_event(uc, mfcr2->t1b, t1b_expired, (void *) (intptr_t) ch);
            mfcr2->mfc_state = MFC_SENT_NOTHING;
        }
        /*endif*/
    }
    /*endif*/
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int send_dnis_digit(uc_t *uc, int ch, int which)
{
    mfcr2_signaling_state_t *mfcr2;
    int forwardsig;
    
    /* Which = 0 for send next digit (n)
             = 1 for send n-1
             = 2 for send n-2
             = 3 for send n-3
             = 4 for start from the beginning
    */
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (which)
    {
        /* We need to hop back and resend some digits, rather than just send
           the next digit */
        if (which <= 3  &&  mfcr2->dnis_cnt > which)
        {
            /* Resend n-1, n-2 or n-3 */
            mfcr2->dnis_cnt -= (which + 1);
        }
        else
        {
            /* Restart from the beginning */
            mfcr2->dnis_cnt = 0;
        }
        /*endif*/        
    }
    /*endif*/
    if (mfcr2->dnis_cnt < mfcr2->dnis_len)
    {
        mfcr2->mfc_state = MFC_SENT_DNIS;
        forwardsig = mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt++];
        if (set_mf_signal(uc, ch, forwardsig))
            return  -1;
        /*endif*/
    }
    else
    {
        if (mfcr2->group_i_end_of_DNIS)
        {
            mfcr2->mfc_state = MFC_SENT_END_OF_DNIS;
            if (set_mf_signal(uc, ch, mfcr2->group_i_end_of_DNIS))
                return  -1;
            /*endif*/
        }
        else
        {
            /* Send nothing, and let the timeout which will occur at the far
               end cause things to continue. */
            clear_outgoing_mf_cycle_timeouts(uc);
            /* We started this call, so we are in control. If the far end doesn't time
               out properly, it may never respond, and we will be stuck forever.
               Use a backstop timer, so things will recover if the far end gets stuck. */
            mfcr2->t1b_timer = uc_schedule_event(uc, mfcr2->t1b, t1b_expired, (void *) (intptr_t) ch);
            mfcr2->mfc_state = MFC_SENT_NOTHING;
        }
        /*endif*/
    }
    /*endif*/
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int send_calling_party_category_request(uc_t *uc)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (mfcr2->group_a_send_calling_party_category_and_switch_to_group_c)
    {
        mfcr2->mfc_group = MFC_BACK_GROUP_C;
        mfcr2->mfc_state = MFC_SENT_CATEGORY_REQUEST;
        if (set_mf_signal(uc, ch, mfcr2->group_a_send_calling_party_category_and_switch_to_group_c))
            return -1;
        /*endif*/
    }   
    else
    {
        mfcr2->mfc_group = MFC_BACK_GROUP_A;
        mfcr2->mfc_state = MFC_SENT_CATEGORY_REQUEST;
        if (set_mf_signal(uc, ch, mfcr2->group_a_send_calling_party_category))
            return -1;
        /*endif*/
    }
    /*endif*/
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void decode_calling_party_category(mfcr2_signaling_state_t *mfcr2, int tone_id)
{
    int i;

    if (mfcr2->call == NULL)
        return;
    /*endif*/
    for (i = 0;  i <= 9;  i++)
    {
        if (tone_id == mfcr2->group_ii[i])
            break;
        /*endif*/
    }
    /*endfor*/
    /* Don't complain about codes we do not recognise. They are probably local
       specials. Just treat them as a valid, but unknown, type. */
    mfcr2->call->callparms.calling_party_category = (i <= 9)  ?  i  :  UC_CALLER_CATEGORY_UNKNOWN;
}
/*- End of function --------------------------------------------------------*/

static void mf_tone_on_event(uc_t *uc, int tone_id)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (mfcr2->mf_rx_signal == tone_id)
        return;
    /*endif*/
    mfcr2->mf_rx_signal = tone_id;
    uc_log(uc,
           UC_LOG_DEBUG_2,
           "     <- %c on  " R2_STATUS_FORMAT "\n",
           tone_id,
           uc->chan[ch].active_rx,
           uc_state2str(uc->chan[ch].state),
           r2_state_to_str(mfcr2),
           mfc_state_to_str(mfcr2->mfc_state));
    if (mfcr2->call->incoming_call)
    {
        if (mfcr2->t3_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->t3_timer);
            mfcr2->t3_timer = -1;
        }
        /*endif*/
        mfcr2->t3_timer = uc_schedule_event(uc, mfcr2->t3, t3_expired, (void *) (intptr_t) ch);
        switch (mfcr2->mfc_group)
        {
        case MFC_BACK_STARTING:
            switch (mfcr2->mfc_state)
            {
            case MFC_SENT_SEIZE_ACK:
                if (R2_SIGI_10 <= tone_id  &&  tone_id <= R2_SIGI_9)
                {
                    /* We are waiting for DNIS, and we just got a digit - good! */
                    mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt++] = tone_id;
                    if (mfcr2->dnis_cnt >= mfcr2->dnis_len)
                    {
                        if (mfcr2->use_dialed_number_events)
                        {
                            /* Ask the application what to do next, and wait. */
                            report_dialed_number(uc, mfcr2->call);
                            break;
                        }
                        /*endif*/
                        mfcr2->dnis_complete = TRUE;
                    }
                    /*endif*/
                    if (mfcr2->dnis_complete  ||  !mfcr2->get_ani_after_dnis)
                    {
                        /* Get the calling party category, and optionally the ANI, now. If we wait
                           until later there might be problems when the system relies on timeout
                           to terminate the DNIS. Also, the dialed number may still be arriving at
                           the far end, and the exchange of ANI digits can utilise this delay nicely. */
                        send_calling_party_category_request(uc);
                    }
                    else
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_A;
                        mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                        if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                if (tone_id == mfcr2->group_i_end_of_DNIS)
                {
                    /* We are waiting for DNIS, and we just got an end of DNIS
                       indication. This is OK - it just means there is no DNIS. */
                    mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt] = '\0';
                    /* This is a hard completion of the DNIS. We cannot get any more digits. */
                    mfcr2->dnis_complete = TRUE;
                    /* Get the calling party category, and optionally the ANI. */
                    send_calling_party_category_request(uc);
                    break;
                }
                /*endif*/
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
                break;
            default:
                /* How did we get in this state? */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
                break;
            }
            /*endswitch*/
            break;
        case MFC_BACK_GROUP_A:
            switch (mfcr2->mfc_state)
            {
            case MFC_SENT_CATEGORY_REQUEST:
                decode_calling_party_category(mfcr2, tone_id);
                if (mfcr2->ani_max_rx_digits > 0)
                {
                    /* We want to get the ANI now. */
                    mfcr2->mfc_state = MFC_SENT_ANI_REQUEST;
                    if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_ANI_digit))
                        break;
                    /*endif*/
                }
                else
                {
                    /* Skip the ANI, and just get the DNIS. */
                    if (!mfcr2->dnis_complete)
                    {
                        mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                        if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                            break;
                        /*endif*/
                    }
                    else
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_B;
                        mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                        if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                }
                /*endif*/
                break;
            case MFC_SENT_ANI_REQUEST:
                if (R2_SIGI_10 <= tone_id  &&  tone_id <= R2_SIGI_9)
                {
                    /* A digit of ANI. */
                    mfcr2->call->callparms.originating_number[mfcr2->ani_cnt++] = tone_id;
                    if (mfcr2->ani_cnt >= mfcr2->ani_len)
                    {
                        /* We have all the ANI we expected. */
                        mfcr2->call->callparms.originating_number[mfcr2->ani_cnt] = '\0';
                        if (mfcr2->dnis_complete)
                        {
                            mfcr2->mfc_group = MFC_BACK_GROUP_B;
                            mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                            if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                                break;
                            /*endif*/
                        }
                        else
                        {
                            /* We still have some DNIS digits to get */
                            mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                            if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                                break;
                            /*endif*/
                        }
                        /*endif*/
                    }
                    else
                    {
                        if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_ANI_digit))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                if (tone_id == mfcr2->group_i_end_of_ANI
                    ||
                    tone_id == mfcr2->group_i_end_of_ANI_restricted)
                {
                    if (tone_id == mfcr2->group_i_end_of_ANI_restricted)
                        mfcr2->ani_restricted = TRUE;
                    /*endif*/
                    /* The ANI is complete. */
                    mfcr2->ani_complete = TRUE;
                    mfcr2->call->callparms.originating_number[mfcr2->ani_cnt] = '\0';
                    if (mfcr2->dnis_complete)
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_B;
                        mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                        if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                            break;
                        /*endif*/
                    }
                    else
                    {
                        /* We still have some DNIS digits to get */
                        mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                        if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
                break;
            case MFC_SENT_DNIS_REQUEST:
                if (R2_SIGI_10 <= tone_id  &&  tone_id <= R2_SIGI_9)
                {
                    /* A digit of DNIS */
                    mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt++] = tone_id;
                    if (mfcr2->dnis_cnt >= mfcr2->dnis_len)
                    {
                        if (mfcr2->use_dialed_number_events)
                        {
                            /* Ask the application what to do next, and wait. */
                            report_dialed_number(uc, mfcr2->call);
                            break;
                        }
                        /*endif*/
                        mfcr2->dnis_complete = TRUE;
                    }
                    /*endif*/
                    if (mfcr2->dnis_complete)
                    {
                        /* We have all the DNIS digits we expected. */
                        mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt] = '\0';
                        if (mfcr2->get_ani_after_dnis)
                        {
                            send_calling_party_category_request(uc);
                        }
                        else
                        {
                            mfcr2->mfc_group = MFC_BACK_GROUP_B;
                            mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                            if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                                break;
                            /*endif*/
                        }
                        /*endif*/
                    }
                    else
                    {
                        if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                if (tone_id == mfcr2->group_i_end_of_DNIS)
                {
                    /* The DNIS is complete. */
                    mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt] = '\0';
                    /* This is a hard completion of the DNIS. We cannot get any more digits. */
                    mfcr2->dnis_complete = TRUE;
                    if (mfcr2->get_ani_after_dnis)
                    {
                        send_calling_party_category_request(uc);
                    }
                    else
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_B;
                        mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                        if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
                break;
            default:
                /* How did we get in this state? */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
                break;
            }
            /*endswitch*/
            break;
        case MFC_BACK_GROUP_B:
            switch (mfcr2->mfc_state)
            {
            case MFC_SENT_CHANGE_TO_GROUP_II:
                /* This indicates a change to group II. */
                switch (tone_id)
                {
                case R2_SIGII_1:
                case R2_SIGII_2:
                case R2_SIGII_3:
                case R2_SIGII_4:
                case R2_SIGII_5:
                case R2_SIGII_6:
                case R2_SIGII_11:
                case R2_SIGII_12:
                case R2_SIGII_13:
                case R2_SIGII_14:
                case R2_SIGII_15:
                    /* The meaning of these signals is nationally defined. */
                    /* So, we now have all the information, and we need to
                       decide what we do with this call - accept or reject! */
                    start_offered(uc, mfcr2->call);
                    break;
                case R2_SIGII_7:
                case R2_SIGII_8:
                case R2_SIGII_9:
                case R2_SIGII_10:
                    /* The meaning of these signals is standardised. */
                    /* So, we now have all the information, and we need to
                       decide what we do with this call - accept or reject! */
                    start_offered(uc, mfcr2->call);
                    break;
                }
                /*endswitch*/
                break;
            default:
                /* How did we get in this state? */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
                break;
            }
            /*endswitch*/
            break;
        case MFC_BACK_GROUP_C:
            switch (mfcr2->mfc_state)
            {
            case MFC_SENT_CATEGORY_REQUEST:
                decode_calling_party_category(mfcr2, tone_id);
                if (mfcr2->ani_max_rx_digits > 0)
                {
                    /* We want to get the ANI now. */
                    mfcr2->mfc_state = MFC_SENT_ANI_REQUEST;
                    if (set_mf_signal(uc, ch, mfcr2->group_c_send_next_ANI_digit))
                        break;
                    /*endif*/
                }
                else
                {
                    /* Skip the ANI, and just get the DNIS. */
                    if (mfcr2->dnis_complete)
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_B;
                        mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                        if (set_mf_signal(uc, ch, mfcr2->group_c_switch_to_group_ii))
                            break;
                        /*endif*/
                    }
                    else
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_A;
                        mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                        if (set_mf_signal(uc, ch, mfcr2->group_c_send_next_DNIS_digit_and_switch_to_group_a))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                }
                /*endif*/
                break;
            case MFC_SENT_ANI_REQUEST:
                if (R2_SIGI_10 <= tone_id  &&  tone_id <= R2_SIGI_9)
                {
                    /* A digit of ANI. */
                    mfcr2->call->callparms.originating_number[mfcr2->ani_cnt++] = tone_id;
                    if (mfcr2->ani_cnt >= mfcr2->ani_len)
                    {
                        /* We have all the ANI we expected. */
                        mfcr2->call->callparms.originating_number[mfcr2->ani_cnt] = '\0';
                        if (mfcr2->dnis_complete)
                        {
                            mfcr2->mfc_group = MFC_BACK_GROUP_B;
                            mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                            if (set_mf_signal(uc, ch, mfcr2->group_c_switch_to_group_ii))
                                break;
                            /*endif*/
                        }
                        else
                        {
                            /* We still have some DNIS digits to get */
                            mfcr2->mfc_group = MFC_BACK_GROUP_A;
                            mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                            if (set_mf_signal(uc, ch, mfcr2->group_c_send_next_DNIS_digit_and_switch_to_group_a))
                                break;
                            /*endif*/
                        }
                        /*endif*/
                    }
                    else
                    {
                        if (set_mf_signal(uc, ch, mfcr2->group_c_send_next_ANI_digit))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                if (tone_id == mfcr2->group_i_end_of_ANI
                    ||
                    tone_id == mfcr2->group_i_end_of_ANI_restricted)
                {
                    if (tone_id == mfcr2->group_i_end_of_ANI_restricted)
                        mfcr2->ani_restricted = TRUE;
                    /*endif*/
                    /* The ANI is complete. */
                    mfcr2->call->callparms.originating_number[mfcr2->ani_cnt] = '\0';
                    if (mfcr2->dnis_complete)
                    {
                        mfcr2->mfc_group = MFC_BACK_GROUP_B;
                        mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                        if (set_mf_signal(uc, ch, mfcr2->group_c_switch_to_group_ii))
                            break;
                        /*endif*/
                    }
                    else
                    {
                        /* We still have some DNIS digits to get */
                        mfcr2->mfc_group = MFC_BACK_GROUP_A;
                        mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                        if (set_mf_signal(uc, ch, mfcr2->group_c_send_next_DNIS_digit_and_switch_to_group_a))
                            break;
                        /*endif*/
                    }
                    /*endif*/
                    break;
                }
                /*endif*/
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
                break;
            default:
                /* How did we get in this state? */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
                break;
            }
            /*endswitch*/
            break;
        default:
            /* How did we get in this state? */
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
            break;
        }
        /*endswitch*/
    }
    else
    {
        set_mf_signal(uc, ch, 0);
    }
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

static void mf_tone_off_event(uc_t *uc)
{
    mfcr2_signaling_state_t *mfcr2;
    int tone_id;
    int ch;
    int i;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (mfcr2->mf_rx_signal == 0)
        return;
    /*endif*/
    tone_id = mfcr2->mf_rx_signal;
    mfcr2->mf_rx_signal = 0;
    if (tone_id)
    {
        uc_log(uc,
               UC_LOG_DEBUG_2,
               "     <- %c off " R2_STATUS_FORMAT "\n",
               tone_id,
               uc->chan[ch].active_rx,
               uc_state2str(uc->chan[ch].state),
               r2_state_to_str(mfcr2),
               mfc_state_to_str(mfcr2->mfc_state));
    }
    else
    {
        uc_log(uc,
               UC_LOG_DEBUG_2,
               "     <-   off " R2_STATUS_FORMAT "\n",
               uc->chan[ch].active_rx,
               uc_state2str(uc->chan[ch].state),
               r2_state_to_str(mfcr2),
               mfc_state_to_str(mfcr2->mfc_state));
    }
    /*endif*/
    if (mfcr2->call->incoming_call)
    {
        if (set_mf_signal(uc, ch, 0))
            return;
        /*endif*/
        switch (mfcr2->mfc_group)
        {
        case MFC_BACK_GROUP_A:
            /* We only need to take action when the tone stops in GROUP_B,
               as that is the end of all the tones. */
            break;
        case MFC_BACK_GROUP_B:
            /* This is the last tone off event from the far end. We must
               respond by sending the answer signal, using the CAS bits. */
            switch (mfcr2->mfc_state)
            {
            case MFC_SENT_CHANGE_TO_GROUP_II:
                /* We expect this to happen, and take no action */
                break;
            case MFC_SENT_ACCEPTED_CHARGE:
            case MFC_SENT_ACCEPTED_NO_CHARGE:
                /* We should now have the group II code. */
                decode_calling_party_category(mfcr2, tone_id);
                /* Set a default reason for final call termination. */
                mfcr2->call->disconnect_reason = UC_CAUSE_NORMAL_CLEARING;
                /* We can now dismiss the tone detector. */
                if (mfcr2->play_ringback_tone)
                {
                    super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_RINGBACK]);
                    select_active_rxtx(uc, ACTIVE_SUPER_TONE);
                }
                else
                {
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                }
                /*endif*/
                /* And now a little delay while the buffers flush */
                mfcr2->answer_guard_timer =
                    uc_schedule_event(uc, mfcr2->answer_guard_time, answer_guard_expired, mfcr2->call);
                break;
            case MFC_SENT_NETWORK_CONGESTION:
                /* Do not react to a tone off event in these circumstances.
                   The far end should be clearing the call. */
                if (mfcr2->play_progress_tones)
                {
                    super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_CONGESTED]);
                    select_active_rxtx(uc, ACTIVE_SUPER_TONE);
                }
                else
                {
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                }
                /*endif*/
                break;
            case MFC_SENT_DEST_BUSY:
                /* Do not react to a tone off event in these circumstances.
                   The far end should be clearing the call. */
                if (mfcr2->play_progress_tones)
                {
                    super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_BUSY]);
                    select_active_rxtx(uc, ACTIVE_SUPER_TONE);
                }
                else
                {
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                }
                /*endif*/
                break;
            case MFC_SENT_UNASSIGNED_NUMBER:
                /* Do not react to a tone off event in these circumstances.
                   The far end should be clearing the call. */
                if (mfcr2->play_progress_tones)
                {
                    super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_NU]);
                    select_active_rxtx(uc, ACTIVE_SUPER_TONE);
                }
                else
                {
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                }
                /*endif*/
                break;
            case MFC_SENT_DEST_OUT_OF_ORDER:
                /* Do not react to a tone off event in these circumstances.
                   The far end should be clearing the call. */
                if (mfcr2->play_progress_tones)
                {
                    super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_BUSY]);
                    select_active_rxtx(uc, ACTIVE_SUPER_TONE);
                }
                else
                {
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                }
                /*endif*/
                break;
            default:
                /* How did we get in this state? */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
                break;
            }
            /*endswitch*/
            break;
        case MFC_BACK_GROUP_C:
            /* We only need to take action when the tone stops in GROUP_B,
               as that is the end of all the tones. */
            break;
        default:
            /* How did we get in this state? */
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
            break;
        }
        /*endswitch*/
    }
    else
    {
        switch (mfcr2->mfc_group)
        {
        case MFC_FWD_GROUP_I:
            if (tone_id == mfcr2->group_a_send_next_DNIS_digit)
            {
                send_dnis_digit(uc, ch, 0);
            }
            else if (tone_id == mfcr2->group_a_send_previous_DNIS_digit)
            {
                send_dnis_digit(uc, ch, 1);
            }
            else if (tone_id == mfcr2->group_a_send_DNIS_digit_n_minus_2)
            {
                send_dnis_digit(uc, ch, 2);
            }
            else if (tone_id == mfcr2->group_a_send_DNIS_digit_n_minus_3)
            {
                send_dnis_digit(uc, ch, 3);
            }
            else if (tone_id == mfcr2->group_a_repeat_all_DNIS_digits)
            {
                send_dnis_digit(uc, ch, 4);
            }
            else if (!mfcr2->calling_party_category_exchanged
                     &&
                     tone_id == mfcr2->group_a_send_calling_party_category)
            {
                /* In some places the get calling party category tone is the same
                   as the get ANI tone. We differentiate by the category coming
                   first. */
                uc_log(uc, UC_LOG_DEBUG_2, "Calling party category 0x%X\n", mfcr2->call->callparms.calling_party_category);
                mfcr2->calling_party_category_exchanged = TRUE;
                mfcr2->mfc_state = MFC_SENT_CATEGORY;
                set_mf_signal(uc, ch, mfcr2->group_ii[mfcr2->call->callparms.calling_party_category]);
            }
            else if (tone_id == mfcr2->group_a_send_next_ANI_digit)
            {
                send_ani_digit(uc, ch, 0);
            }
            else if (tone_id == mfcr2->group_a_network_congestion)
            {
                mfcr2->call->disconnect_reason = UC_CAUSE_NETWORK_CONGESTION;
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_a_immediate_accept)
            {
                mfcr2->await_answer_timer =
                    uc_schedule_event(uc,
                                      mfcr2->max_await_answer,
                                      await_answer_expired,
                                      (void *) mfcr2->call);
                mfcr2->call->callparms.call_type = UC_CALL_TYPE_CHARGED_CALL;
                mfcr2->r2_state = R2_FWD_AWAIT_ANSWER;
                start_alerting(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_a_switch_to_group_ii)
            {
                start_proceeding(uc, mfcr2->call);
                mfcr2->mfc_group = MFC_FWD_GROUP_II;
                mfcr2->mfc_state = MFC_SENT_CATEGORY;
                if (set_mf_signal(uc, ch, mfcr2->group_ii[mfcr2->call->callparms.calling_party_category]))
                    break;
                /*endif*/
                mfcr2->wait_for_group_b_signal_timer =
                    uc_schedule_event(uc,
                                      mfcr2->max_wait_for_group_b_signal,
                                      wait_for_group_b_signal_expired,
                                      (void *) mfcr2->call);
            }
            else if (tone_id == mfcr2->group_a_send_calling_party_category_and_switch_to_group_c)
            {
                uc_log(uc, UC_LOG_DEBUG_2, "Calling party category 0x%X\n", mfcr2->call->callparms.calling_party_category);
                mfcr2->calling_party_category_exchanged = TRUE;
                mfcr2->mfc_group = MFC_FWD_GROUP_III;
                mfcr2->mfc_state = MFC_SENT_CATEGORY;
                set_mf_signal(uc, ch, mfcr2->group_ii[mfcr2->call->callparms.calling_party_category]);
            }
            else
            {
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
            }
            /*endif*/
            break;
        case MFC_FWD_GROUP_II:
            if (mfcr2->wait_for_group_b_signal_timer >= 0)
            {
                uc_schedule_del(uc, mfcr2->wait_for_group_b_signal_timer);
                mfcr2->wait_for_group_b_signal_timer = -1;
            }
            /*endif*/
            if (tone_id == mfcr2->group_b_line_free_charge)
            {
                mfcr2->await_answer_timer =
                    uc_schedule_event(uc,
                                      mfcr2->max_await_answer,
                                      await_answer_expired,
                                      (void *) mfcr2->call);
                mfcr2->call->callparms.call_type = UC_CALL_TYPE_CHARGED_CALL;
                mfcr2->r2_state = R2_FWD_AWAIT_ANSWER;
                start_alerting(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_b_line_free_no_charge)
            {
                mfcr2->await_answer_timer =
                    uc_schedule_event(uc,
                                      mfcr2->max_await_answer,
                                      await_answer_expired,
                                      (void *) mfcr2->call);
                mfcr2->call->callparms.call_type = UC_CALL_TYPE_NO_CHARGE_CALL;
                mfcr2->r2_state = R2_FWD_AWAIT_ANSWER;
                start_alerting(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_b_send_special_information_tone)
            {
                /* Connect the call, with the call type "special information tone" */
                mfcr2->await_answer_timer =
                    uc_schedule_event(uc,
                                      mfcr2->max_await_answer,
                                      await_answer_expired,
                                      (void *) mfcr2->call);
                mfcr2->call->callparms.call_type = UC_CALL_TYPE_SIT;
                mfcr2->r2_state = R2_FWD_AWAIT_ANSWER;
                start_alerting(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_b_user_busy)
            {
                mfcr2->call->disconnect_reason = UC_CAUSE_USER_BUSY;
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_b_line_out_of_order)
            {
                mfcr2->call->disconnect_reason = UC_CAUSE_DEST_OUT_OF_ORDER;
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_b_unassigned_number)
            {
                mfcr2->call->disconnect_reason = UC_CAUSE_UNASSIGNED_NUMBER;
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else if (tone_id == mfcr2->group_b_network_congestion)
            {
                mfcr2->call->disconnect_reason = UC_CAUSE_NETWORK_CONGESTION;
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else
            {
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
            }
            /*endif*/
            break;
        case MFC_FWD_GROUP_III:
            if (tone_id == mfcr2->group_c_send_next_ANI_digit)
            {
                send_ani_digit(uc, ch, 0);
            }
            else if (tone_id == mfcr2->group_c_repeat_all_DNIS_digits_and_switch_to_group_a)
            {
                send_dnis_digit(uc, ch, 4);
                mfcr2->mfc_group = MFC_FWD_GROUP_I;
                mfcr2->mfc_state = MFC_SENT_DNIS;
            }
            else if (tone_id == mfcr2->group_c_send_next_DNIS_digit_and_switch_to_group_a)
            {
                send_dnis_digit(uc, ch, 0);
                mfcr2->mfc_group = MFC_FWD_GROUP_I;
                mfcr2->mfc_state = MFC_SENT_DNIS;
            }
            else if (tone_id == mfcr2->group_c_send_previous_DNIS_digit_and_switch_to_group_a)
            {
                send_dnis_digit(uc, ch, 1);
                mfcr2->mfc_group = MFC_FWD_GROUP_I;
                mfcr2->mfc_state = MFC_SENT_DNIS;
            }
            else if (tone_id == mfcr2->group_c_switch_to_group_ii)
            {
                mfcr2->wait_for_group_b_signal_timer =
                    uc_schedule_event(uc,
                                      mfcr2->max_wait_for_group_b_signal,
                                      wait_for_group_b_signal_expired,
                                      (void *) mfcr2->call);
                start_proceeding(uc, mfcr2->call);
                mfcr2->mfc_group = MFC_FWD_GROUP_II;
                mfcr2->mfc_state = MFC_SENT_CATEGORY;
                if (set_mf_signal(uc, ch, mfcr2->group_ii[mfcr2->call->callparms.calling_party_category]))
                    break;
                /*endif*/
            }
            else
            {
                /* We were not expecting this, whatever it is! */
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
            }
            /*endif*/
            break;
        default:
            if (mfcr2->r2_state == R2_FWD_ANSWERED_BEFORE_MFC_COMPLETE)
            {
                if (mfcr2->wait_for_group_b_signal_timer >= 0)
                {
                    uc_schedule_del(uc, mfcr2->wait_for_group_b_signal_timer);
                    mfcr2->wait_for_group_b_signal_timer = -1;
                }
                /*endif*/
                if (tone_id == mfcr2->group_b_line_free_charge)
                {
                    /* This should come before the CAS ANSWER signal. But timing
                    overlap may mean we see the CAS answer signal first. */
                    mfcr2->call->callparms.call_type = UC_CALL_TYPE_CHARGED_CALL;
                    mfcr2->r2_state = R2_FWD_ANSWERED;
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                    start_connected(uc, mfcr2->call);
                }
                else if (tone_id == mfcr2->group_b_line_free_no_charge)
                {
                    /* This should come before the CAS ANSWER signal. But timing
                    overlap may mean we see the CAS answer signal first. */
                    mfcr2->call->callparms.call_type = UC_CALL_TYPE_NO_CHARGE_CALL;
                    mfcr2->r2_state = R2_FWD_ANSWERED;
                    select_active_rxtx(uc, ACTIVE_SPEECH);
                    start_connected(uc, mfcr2->call);
                }
                else
                {
                    /* We were not expecting this, whatever it is! */
                    protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL);
                }
                /*endif*/
                break;
            }
            /*endif*/
            /* How did we get in this state? */
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
            break;
        }
        /*endswitch*/
    }
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

static void abcd_signaling_event(uc_t *uc, int ch)
{
    int abcd;
    int masked_abcd;
    int changes;
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    if (check_abcd(uc, ch, &abcd) != UC_RET_OK)
        return;
    /*endif*/
    uc_log(uc,
           UC_LOG_DEBUG_2,
           "     <- %d%d%d%d  " R2_STATUS_FORMAT "\n",
           (abcd >> 3) & 1, (abcd >> 2) & 1, (abcd >> 1) & 1, abcd & 1,
           uc->chan[ch].active_rx,
           uc_state2str(uc->chan[ch].state),
           r2_state_to_str(mfcr2),
           mfc_state_to_str(mfcr2->mfc_state));
    changes = mfcr2->current_abcd_in ^ abcd;
    mfcr2->current_abcd_in = abcd;
    /* TODO: deal with charging pulses, and other things, which occur in the C and D
             bits in some places. */
    /* TODO: deal with charging pulses, which occur on the A or B bits during the connected
             phase in some places. */
    /* The bits which changed might not be the R2 signaling bits. */
    if ((changes & mfcr2->abcd_mask) == 0)
        return;
    /*endif*/
    masked_abcd = abcd & mfcr2->abcd_mask;
    switch (uc->chan[ch].state)
    {
    case UC_STATE_NULL:
        /* In reality, for polling mode, this branch is never taken. */
        if (masked_abcd == mfcr2->abcd_idle)
        {
            uc_log(uc, UC_LOG_WARNING, "Got CAS idle in UC_STATE_NULL\n");
            uc->chan[ch].state = UC_STATE_IDLE;
            mfcr2->r2_state = R2_IDLE;
            mfcr2->mfc_group = 0;
            mfcr2->mfc_state = 0;
            mfcr2->calling_party_category_exchanged = FALSE;
            select_active_rxtx(uc, ACTIVE_SPEECH);
        }
        else
        {
            /* In initial state, any signal could be received. */
        }
        /*endif*/
        return;
    case UC_STATE_IDLE:
        if (masked_abcd == mfcr2->abcd_idle)
        {
            if (set_abcd_signal(uc, ch, mfcr2->abcd_idle))
                return;
            /*endif*/
        }
        else if (masked_abcd == mfcr2->fwd_abcd_seize)
        {
            start_detected(uc, ch);
        }
        else if (masked_abcd == mfcr2->abcd_blocking)
        {
            uc->chan[ch].state = UC_STATE_BLOCKED;
            mfcr2->far_blocked = TRUE;
            select_active_rxtx(uc, ACTIVE_SPEECH);
            ev.e = UC_EVENT_FARBLOCKED;
            ev.gen.channel = ch;
            uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
        }
        else
        {
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
        }
        /*endif*/
        return;
    case UC_STATE_BLOCKED:
        if (masked_abcd == mfcr2->abcd_idle)
        {
            /* Start a timeout to ensure we do not seize an unblocked trunk too
               agressively. */
            mfcr2->far_unblocking_timer =
                uc_schedule_event(uc, mfcr2->blocking_release_time, far_unblocking_expired, (void *) (intptr_t) ch);
        }
        else if (masked_abcd == mfcr2->fwd_abcd_seize)
        {
            /* We must have got a seize before we expired an unblocking timer.
               Force the end of the blocking timer, and accept the call. */
            if (mfcr2->far_unblocking_timer >= 0)
            {
                uc_schedule_del(uc, mfcr2->far_unblocking_timer);
                mfcr2->far_unblocking_timer = -1;
                far_unblocking_expired(uc, (void *) (intptr_t) ch);
            }
            /*endif*/
            if (mfcr2->local_unblocking_timer >= 0)
            {
                uc_schedule_del(uc, mfcr2->local_unblocking_timer);
                mfcr2->local_unblocking_timer = -1;
                local_unblocking_expired(uc, (void *) (intptr_t) ch);
            }
            /*endif*/
            start_detected(uc, ch);
        }
        else if (masked_abcd == mfcr2->abcd_blocking)
        {
            /* This could happen if the line is blocked at startup */
            uc->chan[ch].state = UC_STATE_BLOCKED;
            mfcr2->far_blocked = TRUE;
            select_active_rxtx(uc, ACTIVE_SPEECH);
            ev.e = UC_EVENT_FARBLOCKED;
            ev.gen.channel = ch;
            uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
        }
        else
        {
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
        }
        /*endif*/
        return;
    }
    /*endswitch*/

    /* There must be a call in progress */
    if (mfcr2->call == NULL)
    {
        /* This is bad. Try to avoid things falling apart, by doing a protocol recovery.
           However, if we are here it is not a protocol problem. Its a bug. */
        uc_log(uc, UC_LOG_ERROR, "!!!! ABCD in call state 0x%X with no call structure\n", uc->chan[ch].state);
        protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
        return;
    }
    /*endif*/
    if (mfcr2->call->incoming_call)
    {
        switch (mfcr2->r2_state)
        {
        case R2_BACK_SEIZE_ACK:
        case R2_BACK_ANSWER:
            if (masked_abcd == mfcr2->fwd_abcd_clear_fwd)
            {
                mfcr2->r2_state = R2_BACK_CLEAR_FWD;
                start_far_disconnected(uc, mfcr2->call);
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_BACK_CLEAR_BACK:
            if (masked_abcd == mfcr2->fwd_abcd_clear_fwd)
                start_call_disconnected(uc, mfcr2->call);
            else
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            /*endif*/
            break;
        case R2_BACK_RELEASE_GUARD:
            if (masked_abcd == mfcr2->fwd_abcd_seize)
            {
                /* An enthusiastic far end has seized for a new call, while we
                   were still timing out the last one. Make like the timeout
                   occured, and carry on with the new call. */
                if (mfcr2->release_guard_timer >= 0)
                    uc_schedule_del(uc, mfcr2->release_guard_timer);
                /*endif*/
                release_guard_expired(uc, (void *) mfcr2->call);

                /* .... and now the new call begins. */
                start_detected(uc, ch);
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        default:
            /* Unknown phase. Handle as a protocol error. */
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
            break;
        }
        /*endswitch*/
    }
    else
    {
        switch (mfcr2->r2_state)
        {
        case R2_FWD_SEIZE:
            if (masked_abcd == mfcr2->back_abcd_seize_ack)
            {
                if (mfcr2->seize_ack_timer >= 0)
                {
                    uc_schedule_del(uc, mfcr2->seize_ack_timer);
                    mfcr2->seize_ack_timer = -1;
                }
                /*endif*/
                /* The other side's tone detector should now be listening.
                   We have not started sending tone, so we cannot have missed
                   any incoming tone if we only enable the detector now. Enabling
                   the detector runs the risk of picking up something from the
                   speech path, before the other end muted it. MF6 is not very
                   speech immune. */
                select_active_rxtx(uc, ACTIVE_MF6);
                /* Start the MF phase by sending the DNIS. The other
                   end may change from DNIS by the signals it returns,
                   but we always start with the DNIS. */
                mfcr2->r2_state = R2_FWD_SEIZE_ACK;
                mfcr2->mfc_group = MFC_FWD_GROUP_I;
                mfcr2->dnis_cnt = 0;
                mfcr2->ani_cnt = 0;
                send_dnis_digit(uc, ch, 0);
            }
            else if (masked_abcd == mfcr2->fwd_abcd_seize)
            {
                /* Call collision! We should clear, and abandon this call attempt. */
                mfcr2->call->disconnect_reason = UC_CAUSE_NETWORK_CONGESTION;
                /* Provoke the application to drop the call. */
                start_far_disconnected(uc, mfcr2->call);
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_FWD_SEIZE_ACK:
            if (masked_abcd == mfcr2->back_abcd_answer)
            {
                /* Timing overlaps might legitimately put us in this
                   position. */
                mfcr2->r2_state = R2_FWD_ANSWERED_BEFORE_MFC_COMPLETE;
            }
            else if (masked_abcd == mfcr2->back_abcd_clear_back)
            {
                /* This isn't really after answer, but needs to be treated as such.
                   In fact, this condition cannot generally occur. */
                mfcr2->r2_state = R2_FWD_CLEAR_BACK_AFTER_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else if (masked_abcd == mfcr2->abcd_idle)
            {
                /* The far end has abandoned the call for some reason. */
                /* TODO: handle this cleanly */
                mfcr2->r2_state = R2_FWD_CLEAR_BACK_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_FWD_AWAIT_ANSWER:
            if (masked_abcd == mfcr2->back_abcd_answer)
            {
                mfcr2->r2_state = R2_FWD_ANSWERED;
                start_connected(uc, mfcr2->call);
            }
            else if (masked_abcd == mfcr2->back_abcd_clear_back)
            {
                /* This isn't really after answer, but needs to be treated as such.
                   In fact, this condition cannot generally occur. */
                mfcr2->r2_state = R2_FWD_CLEAR_BACK_AFTER_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else if (masked_abcd == mfcr2->abcd_idle)
            {
                /* The far end has abandoned the call before we gave up waiting for answer. */
                /* TODO: handle this cleanly */
                mfcr2->r2_state = R2_FWD_CLEAR_BACK_BEFORE_ANSWER;
                start_far_disconnected(uc, mfcr2->call);
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_FWD_ANSWERED_BEFORE_MFC_COMPLETE:
        case R2_FWD_ANSWERED:
            if (masked_abcd == mfcr2->back_abcd_clear_back)
            {
                mfcr2->r2_state = R2_FWD_CLEAR_BACK_AFTER_ANSWER;
                if (mfcr2->clear_back_persistence_check)
                {
                    /* We need to apply a persistence check to separate real clearbacks
                       from metering pulses. */
                    mfcr2->clear_back_persistence_timer =
                        uc_schedule_event(uc, mfcr2->clear_back_persistence_check, clear_back_persistence_expired, mfcr2->call);
                }
                else
                {
                    start_far_disconnected(uc, mfcr2->call);
                }
                /*endif*/
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_FWD_CLEAR_BACK_BEFORE_ANSWER:
            /* There should be no CAS transitions in this state. */
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            break;
        case R2_FWD_CLEAR_BACK_AFTER_ANSWER:
            if (mfcr2->clear_back_persistence_timer >= 0)
            {
                /* We have not yet confirmed the clear back condition. */
                uc_schedule_del(uc, mfcr2->clear_back_persistence_timer);
                mfcr2->clear_back_persistence_timer = -1;
                if (masked_abcd == mfcr2->back_abcd_answer)
                {
                    /* It must have been a metering pulse. Return to the answered state. */
                    mfcr2->r2_state = R2_FWD_ANSWERED;
                    break;
                }
                /*endif*/
            }
            /*endif*/
            /* If they applied CLEAR BACK, there should be no more CAS transitions
               until we take some action. */
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            break;
        case R2_FWD_CLEAR_FWD_BEFORE_SEIZE_ACK:
            if (masked_abcd == mfcr2->back_abcd_seize_ack)
            {
                if (mfcr2->seize_ack_timer >= 0)
                {
                    uc_schedule_del(uc, mfcr2->seize_ack_timer);
                    mfcr2->seize_ack_timer = -1;
                }
                /*endif*/
                /* It is now safe to send the pending CLEAR FWD. */
                if (start_clear_fwd(uc))
                    return;
                /*endif*/
            }
            else if (masked_abcd == mfcr2->fwd_abcd_seize)
            {
                /* Call collision! We should clear, and abandon this call attempt. */
                mfcr2->call->disconnect_reason = UC_CAUSE_NETWORK_CONGESTION;
                if (start_clear_fwd(uc))
                    return;
                /*endif*/
                /* The application has already dropped this call, so just wait
                   until the far end reacts and disconnects. */
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_FWD_CLEAR_FWD_BEFORE_ANSWER:
            if (masked_abcd == mfcr2->abcd_idle)
                start_idle(uc, mfcr2->call);
            else
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            /*endif*/
            break;
        case R2_FWD_CLEAR_FWD_AFTER_ANSWER:
            if (masked_abcd == mfcr2->abcd_idle)
            {
                start_call_disconnected(uc, mfcr2->call);
            }
            else if (masked_abcd == mfcr2->back_abcd_clear_back)
            {
                /* If they applied CLEAR BACK just before they saw our CLEAR FWD
                   this could legitimately occur. */
                start_far_disconnected(uc, mfcr2->call);
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_AFTER_CLEAR_BACK;
            }
            else
            {
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            }
            /*endif*/
            break;
        case R2_FWD_CLEAR_FWD_AFTER_CLEAR_BACK:
            if (masked_abcd == mfcr2->abcd_idle)
                start_call_disconnected(uc, mfcr2->call);
            else
                protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS);
            /*endif*/
            break;
        default:
            protocol_error_recovery(uc, ch, MFCR2_PROTOCOL_FAIL_INVALID_STATE);
            break;
        }
        /*endswitch*/
    }
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

static uc_event_t *check_event(uc_t *uc)
{
    uint8_t buf[1024];
    int16_t pcm_buf[1024];
    int res;
    int len;
    int xlen;
    int maxlen;
    int i;
    int x;
    int changes;
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    int ch;
    int alarms;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    for (;;)
    {
        //x = ZT_IOMUX_SIGEVENT | ZT_IOMUX_READ | ZT_IOMUX_WRITE | ZT_IOMUX_WRITEEMPTY | ZT_IOMUX_NOWAIT;
        x = ZT_IOMUX_SIGEVENT | ZT_IOMUX_READ | ZT_IOMUX_NOWAIT;
        switch (uc->chan[ch].active_tx)
        {
        case ACTIVE_SPEECH:
            if (uc->chan[ch].channel_write)
                x |= ZT_IOMUX_WRITE;
            /*endif*/
            break;
        case ACTIVE_MF6:
            if (mfcr2->mf_tx_signal)
                x |= ZT_IOMUX_WRITE;
            /*endif*/
            break;
        case ACTIVE_SUPER_TONE:
            x |= ZT_IOMUX_WRITE;
            break;
        }
        /*endif*/
        if ((res = ioctl(uc->chan[ch].handle, ZT_IOMUX, &x)))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            break;
        }
        /*endif*/
        if (x == 0)
            break;
        /*endif*/
        if ((x & ZT_IOMUX_SIGEVENT))
        {
            //uc_log(uc, UC_LOG_FLOW, "Event\n");
            res = ioctl(uc->chan[ch].handle, ZT_GETEVENT, &x);
            if (res == 0  &&  x != 0)
            {
                switch (x)
                {
                case ZT_EVENT_BITSCHANGED:
                    abcd_signaling_event(uc, ch);
                    break;
                case ZT_EVENT_ALARM:
                case ZT_EVENT_NOALARM:
                    if (check_alarms(uc, ch, &alarms) != UC_RET_OK)
                    {
                        /* TODO: */
                    }
                    /*endif*/
                    changes = mfcr2->alarms ^ alarms;
                    ev.alarm.e = UC_EVENT_ALARM;
                    ev.alarm.channel = ch;
                    ev.alarm.raised = changes & alarms;
                    ev.alarm.cleared = changes & alarms;
                    mfcr2->alarms = alarms;
                    uc_report_event(uc, &ev, sizeof(ev));
                    break;
                default:
                    if (uc->chan[ch].channel_error)
                        uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
                    /*endif*/
                    break;
                }
                /*endswitch*/
            }
            /*endif*/
            continue;
        }
        /*endif*/
        if ((x & ZT_IOMUX_READ))
        {
            //uc_log(uc, UC_LOG_FLOW, "Read - %d\n", uc->chan[ch].handle);
            if ((res = read(uc->chan[ch].handle, buf, sizeof(buf))) < 0)
            {
                if (errno == ELAST)
                {
                    /* This means there is some out of band stuff to deal with - alarms,
                       CAS bits signaling changes, etc. */
                }
                else if (errno != EAGAIN)
                {
                    if (uc->chan[ch].channel_error)
                        uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
                    /*endif*/
                }
                /*endif*/
                continue;
            }
            /*endif*/
            switch (uc->chan[ch].active_rx)
            {
            case ACTIVE_MF6:
                /* Do the tone detection */
                switch (uc->chan[ch].api_codec)
                {
                case UC_CODEC_LINEAR16:
#if defined(AUDIO_LOG)
                    if (mfcr2->audio_rx_log > 0)
                        write(mfcr2->audio_rx_log, buf, res);
                    /*endif*/
#endif
                    x = r2_mf_rx(&mfcr2->mf_rx_state, (int16_t *) buf, res >> 1);
                    break;
                case UC_CODEC_ALAW:
#if defined(AUDIO_LOG)
                    if (mfcr2->audio_rx_log > 0)
                        write(mfcr2->audio_rx_log, buf, res);
                    /*endif*/
#endif
                    for (i = 0;  i < res;  i++)
                        pcm_buf[i] = alaw_to_linear(buf[i]);
                    /*endfor*/
                    x = r2_mf_rx(&mfcr2->mf_rx_state, pcm_buf, res);
                    break;
                case UC_CODEC_ULAW:
#if defined(AUDIO_LOG)
                    if (mfcr2->audio_rx_log > 0)
                        write(mfcr2->audio_rx_log, buf, res);
                    /*endif*/
#endif
                    for (i = 0;  i < res;  i++)
                        pcm_buf[i] = ulaw_to_linear(buf[i]);
                    /*endfor*/
                    x = r2_mf_rx(&mfcr2->mf_rx_state, pcm_buf, res);
                    break;
                default:
                    uc_log(uc, UC_LOG_WARNING, "Unknown codec %d\n", uc->chan[ch].api_codec);
                    x = -1;
                    break;
                }
                /*endswitch*/
                if (x != -1)
                {
                    if (x)
                        mf_tone_on_event(uc, x);
                    else
                        mf_tone_off_event(uc);
                    /*endif*/
                }
                /*endif*/
                break;
            case ACTIVE_SPEECH:
                if (uc->chan[ch].channel_read)
                {
                    if (uc->chan[ch].channel_read_codec)
                        res = uc->chan[ch].channel_read_codec(uc, ch, uc->chan[ch].channel_read_codec_user_data, buf, res);
                    /*endif*/
                    uc->chan[ch].channel_read(uc, ch, uc->chan[ch].channel_read_user_data, buf, res);
                }
                /*endif*/
                break;
            }
            /*endswitch*/
            continue;
        }
        /*endif*/
        if ((x & ZT_IOMUX_WRITE))
        {
            //uc_log(uc, UC_LOG_FLOW, "Write\n");
            switch (uc->chan[ch].active_tx)
            {
            case ACTIVE_MF6:
                if (mfcr2->mf_tx_signal == 0)
                    break;
                /*endif*/
                /* Fall through */
            case ACTIVE_SUPER_TONE:
                maxlen = uc->chan[ch].audio_bufsize;
                if (uc->chan[ch].active_tx == ACTIVE_MF6)
                    len = r2_mf_tx(&mfcr2->mf_tx_state, pcm_buf,  maxlen);
                else
                    len = super_tone_tx(&mfcr2->super_tone_tx_state, pcm_buf, maxlen);
                /*endif*/
                if (len == 0)
                    break;
                /*endif*/
                switch (uc->chan[ch].api_codec)
                {
                case UC_CODEC_LINEAR16:
                    len *= 2;
#if defined(AUDIO_LOG)
                    if (mfcr2->audio_tx_log > 0)
                        write(mfcr2->audio_tx_log, pcm_buf, len);
                    /*endif*/
#endif
                    xlen = write(uc->chan[ch].handle, pcm_buf, len);
                    break;
                case UC_CODEC_ALAW:
                    for (i = 0;  i < len;  i++)
                        buf[i] = linear_to_alaw(pcm_buf[i]);
                    /*endfor*/
#if defined(AUDIO_LOG)
                    if (mfcr2->audio_tx_log > 0)
                        write(mfcr2->audio_tx_log, buf, len);
                    /*endif*/
#endif
                    xlen = write(uc->chan[ch].handle, buf, len);
                    break;
                case UC_CODEC_ULAW:
                    for (i = 0;  i < len;  i++)
                        buf[i] = linear_to_ulaw(pcm_buf[i]);
                    /*endfor*/
#if defined(AUDIO_LOG)
                    if (mfcr2->audio_tx_log > 0)
                        write(mfcr2->audio_tx_log, buf, len);
                    /*endif*/
#endif
                    xlen = write(uc->chan[ch].handle, buf, len);
                    break;
                default:
                    uc_log(uc, UC_LOG_WARNING, "Unknown codec %d\n", uc->chan[ch].api_codec);
                    break;
                }
                /*endswitch*/
                if (xlen != len)
                {
                    if (uc->chan[ch].channel_error)
                        uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
                    /*endif*/
                }
                /*endif*/
                break;
            case ACTIVE_SPEECH:
                if (uc->chan[ch].channel_write)
                {
                    maxlen = uc->chan[ch].audio_bufsize;
                    len = uc->chan[ch].channel_write(uc, ch, uc->chan[ch].channel_write_user_data, buf, maxlen);
                    if (len > 0)
                    {
                        if (uc->chan[ch].channel_write_codec)
                            len = uc->chan[ch].channel_write_codec(uc, ch, uc->chan[ch].channel_write_codec_user_data, buf, len);
                        /*endif*/
                        xlen = write(uc->chan[ch].handle, buf, len);
                        if (xlen != len)
                        {
                            if (uc->chan[ch].channel_error)
                                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
                            /*endif*/
                        }
                        /*endif*/
                    }
                    /*endif*/
                }
                /*endif*/
                break;
            }
            /*endswitch*/
            continue;
        }
        /*endif*/
        if ((x & ZT_IOMUX_WRITEEMPTY))
        {
            //uc_log(uc, UC_LOG_FLOW, "Write empty\n");
            continue;
        }
        /*endif*/
    }
    /*endfor*/
    return NULL;
}
/*- End of function --------------------------------------------------------*/

static void load_r2_parameter_set(mfcr2_signaling_state_t *mfcr2, int protocol, int outgoing_calls_allowed)
{
    mfcr2->outgoing_calls_allowed = outgoing_calls_allowed;

    /* Set the standard values for everything first */
    r2_mf_tx_init(&mfcr2->mf_tx_state, FALSE);
    r2_mf_rx_init(&mfcr2->mf_rx_state, TRUE);

    mfcr2->abcd_mask = 0xC;
    mfcr2->base_abcd_bits = 0x1;

    mfcr2->abcd_idle = 0x8;
    mfcr2->abcd_blocking = 0xC;

    mfcr2->fwd_abcd_seize = 0x0;
    mfcr2->fwd_abcd_clear_fwd = 0x8;

    mfcr2->back_abcd_seize_ack = 0xC;
    mfcr2->back_abcd_answer = 0x4;
    mfcr2->back_abcd_clear_back = 0xC;

    mfcr2->abcd_billing_pulse = 0;
    mfcr2->billing_pulse_duration = 0;

    /* Group I signals. */
    mfcr2->group_i_end_of_DNIS = R2_SIGI_15;
    mfcr2->group_i_end_of_ANI = R2_SIGI_15;
    mfcr2->group_i_end_of_ANI_restricted = 0;

    /* Group II signals. */
    mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_SUBSCRIBER_CALL] = R2_SIGII_1;
    mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_PRIORITY_SUBSCRIBER_CALL] = R2_SIGII_2;
    mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_MAINTENANCE_CALL] = R2_SIGII_3;
    mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_OPERATOR_CALL] = R2_SIGII_5;
    mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_DATA_CALL] = R2_SIGII_6;
    mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_PAYPHONE_CALL] = 0;
    mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_SUBSCRIBER_CALL] = R2_SIGII_7;
    mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_PRIORITY_SUBSCRIBER_CALL] = R2_SIGII_9;
    mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_OPERATOR_CALL] = R2_SIGII_10;
    mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_DATA_CALL] = R2_SIGII_8;

    /* Group A signals. */
    mfcr2->group_a_send_next_DNIS_digit = R2_SIGA_1;
    mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_2;
    mfcr2->group_a_switch_to_group_ii = R2_SIGA_3;
    mfcr2->group_a_network_congestion = R2_SIGA_4;
    mfcr2->group_a_send_calling_party_category = R2_SIGA_5;
    mfcr2->group_a_immediate_accept = R2_SIGA_6;
    mfcr2->group_a_send_DNIS_digit_n_minus_2 = R2_SIGA_7;
    mfcr2->group_a_send_DNIS_digit_n_minus_3 = R2_SIGA_8;
    mfcr2->group_a_repeat_all_DNIS_digits = 0;
    mfcr2->group_a_send_next_ANI_digit = R2_SIGA_5;
    mfcr2->group_a_send_calling_party_category_and_switch_to_group_c = 0;
    
    /* Group B signals. */
    mfcr2->group_b_send_special_information_tone = R2_SIGB_2;
    mfcr2->group_b_user_busy = R2_SIGB_3;
    mfcr2->group_b_network_congestion = R2_SIGB_4;
    mfcr2->group_b_unassigned_number = R2_SIGB_5;
    mfcr2->group_b_line_free_charge = R2_SIGB_6;
    mfcr2->group_b_line_free_no_charge = R2_SIGB_7;
    mfcr2->group_b_line_out_of_order = R2_SIGB_8;

    /* Group C signals. */
    mfcr2->group_c_send_next_ANI_digit = 0;
    mfcr2->group_c_repeat_all_DNIS_digits_and_switch_to_group_a = 0;
    mfcr2->group_c_send_next_DNIS_digit_and_switch_to_group_a = 0;
    mfcr2->group_c_send_previous_DNIS_digit_and_switch_to_group_a = 0;

    mfcr2->dnis_max_rx_digits = 7;
    mfcr2->ani_max_rx_digits = UC_MAXPHONENUMLEN;

    mfcr2->t1 = DEFAULT_T1;
    mfcr2->t1a = DEFAULT_T1A;
    mfcr2->t1b = DEFAULT_T1B;
    mfcr2->t2 = DEFAULT_T2;
    mfcr2->t3 = DEFAULT_T3;
    
    mfcr2->max_wait_for_group_b_signal = DEFAULT_MAX_WAIT_FOR_GROUP_B_SIGNAL;
    mfcr2->max_await_answer = DEFAULT_MAX_AWAIT_ANSWER;
    mfcr2->blocking_release_time = DEFAULT_BLOCKING_RELEASE_TIME;
    mfcr2->answer_guard_time = DEFAULT_ANSWER_GUARD_TIME;
    mfcr2->release_guard_time = DEFAULT_RELEASE_GUARD_TIME;
    mfcr2->max_seize_ack_wait = DEFAULT_MAX_SEIZE_ACK_WAIT;

    mfcr2->inbound_cas_persistence_check = 20;
    mfcr2->outbound_cas_persistence_check = 20;
    mfcr2->clear_back_persistence_check = 0;

    /* For several variants of MFC/R2, the forward I-15 tone signals the
       end of ANI digits, but not the end of DNIS digits. If the inbound
       register expects a variable number of inbound digits, and it
       requests one digit more than the outbound register has available,
       the outbound side remains silent until the inbound side times out.
       Then the inbound side pulses the A-3 tone to resume the compelled
       sequence, and switches to the group II/B compelled signals. */
    mfcr2->group_b_on_DNIS_timeout = FALSE;

    /* Now make any specific modifications for the R2 variant in use */
    switch (protocol)
    {
    case MFCR2_PROT_ALGERIA:
        mfcr2->dnis_max_rx_digits = 9;
        /* TODO: */
        break;
    case MFCR2_PROT_ARGENTINA:
        mfcr2->group_i_end_of_DNIS = 0;
        mfcr2->group_i_end_of_ANI_restricted = R2_SIGI_12;

        mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_9;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_10;

        mfcr2->group_b_on_DNIS_timeout = TRUE;
        mfcr2->t3 = 5000;
 
        /* There mey be metering pulses on bit A, demanding the timed detection
           of release. The line code that the outbound side sees during a metering
           pulse to be the same as the line code that signals the clear back
           condition. This line code must persist for at least 400 ms to be a
           valid clear back condition. */
        break;
    case MFCR2_PROT_BAHRAIN:
        /* As per ITU spec. */
        break;
    case MFCR2_PROT_BOLIVIA:
        mfcr2->group_a_send_DNIS_digit_n_minus_3 = 0;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_9;
        mfcr2->group_i_end_of_ANI_restricted = R2_SIGI_12;
        break;
    case MFCR2_PROT_BRAZIL:
        mfcr2->group_i_end_of_DNIS = 0;
        mfcr2->group_i_end_of_ANI_restricted = R2_SIGI_12;

        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        mfcr2->group_a_immediate_accept = 0;
        mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_9;

        mfcr2->group_b_send_special_information_tone = R2_SIGB_6;
        mfcr2->group_b_unassigned_number = R2_SIGB_7;
        mfcr2->group_b_line_free_charge = R2_SIGB_1;
        mfcr2->group_b_line_free_no_charge = R2_SIGB_5;

        mfcr2->group_b_on_DNIS_timeout = TRUE;
        mfcr2->t3 = 5000;

        /* Some switches have a provision for a "reanswer" signal with
           which an inbound terminal equipment can automatically reject
           all collect calls. To do this, the equipment pulses a clear
           back line code, immediately after signaling the answer line
           code to connect a call. If this pulse is sent to the switch,
           and the call is a collect call, the switch clears the call
           back and then clears the line forward. */
        break;
    case MFCR2_PROT_CHILE:
        /* As per ITU spec., plus billing pulses on bit C */
        break;
    case MFCR2_PROT_CHINA:
    case MFCR2_PROT_THAILAND:
        /* China and Thailand set the CAS C bit, which most countries do not. */
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_MAINTENANCE_CALL] = R2_SIGII_6;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_DATA_CALL] = R2_SIGII_4;

        mfcr2->base_abcd_bits = 0x3;

        mfcr2->group_i_end_of_DNIS = 0;

        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_send_previous_DNIS_digit = 0;
        mfcr2->group_a_send_DNIS_digit_n_minus_2 = 0;
        mfcr2->group_a_send_DNIS_digit_n_minus_3 = 0;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_1;
    
        mfcr2->group_b_user_busy = R2_SIGB_2;
        mfcr2->group_b_line_free_charge = R2_SIGB_1;

        mfcr2->group_b_on_DNIS_timeout = TRUE;
        mfcr2->t3 = 5000;
        break;
    case MFCR2_PROT_COLOMBIA_LAND:
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_immediate_accept = 0;
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_1;
        break;
    case MFCR2_PROT_COLOMBIA_CELL:
        mfcr2->group_i_end_of_DNIS = 0;

        mfcr2->group_a_send_previous_DNIS_digit = 0;
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_immediate_accept = 0;
        mfcr2->group_a_send_DNIS_digit_n_minus_2 = 0;
        mfcr2->group_a_send_DNIS_digit_n_minus_3 = 0;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_1;

        mfcr2->group_b_on_DNIS_timeout = TRUE;
        mfcr2->t3 = 5000;
        break;
    case MFCR2_PROT_CZECH:
        /* The compelled timer T3 (inbound compelled whole-cycle timer) is set to 5s
           (the ITU value is 15s). For a slow compelled sequence, the inbound
           side will time out before the outbound side, initiating clearing of the
           connection. */
        mfcr2->t3 = 5000;
        break;
    case MFCR2_PROT_DEMO_CONGO:
        /* As per ITU spec. */
        break;
    case MFCR2_PROT_EGYPT:
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_1;
        /* TODO: */
        break;
    case MFCR2_PROT_GHANA:
        /* TODO: */
        break;
    case MFCR2_PROT_HONDURAS:
        /* As per ITU spec. */
        break;
    case MFCR2_PROT_INDIA:
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_DATA_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_PAYPHONE_CALL] = R2_SIGII_6;

        mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_9;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_4;
        break;
    case MFCR2_PROT_INDONESIA:
        mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_8;
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_immediate_accept = R2_SIGA_5;
        mfcr2->group_a_send_DNIS_digit_n_minus_2 = R2_SIGA_9;
        mfcr2->group_a_send_DNIS_digit_n_minus_3 = 0;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        break;
    case MFCR2_PROT_IRAQ:
        /* As per ITU spec. */
        break;
    case MFCR2_PROT_KOREA:
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_9;
        break;
    case MFCR2_PROT_KUWAIT:
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_2;
        mfcr2->group_b_user_busy = R2_SIGB_2;
        mfcr2->group_b_line_free_charge = R2_SIGB_5;
        mfcr2->max_seize_ack_wait = 4000;
        /* TODO: */
        break;
    case MFCR2_PROT_MALAYSIA:
        mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_8;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_immediate_accept = 0;
        mfcr2->group_a_send_DNIS_digit_n_minus_2 = R2_SIGA_9;
        mfcr2->group_a_send_DNIS_digit_n_minus_3 = 0;
        break;
    case MFCR2_PROT_MEXICO:
        mfcr2->group_i_end_of_DNIS = 0;

        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        mfcr2->group_a_send_calling_party_category = 0;
        mfcr2->group_a_send_next_ANI_digit = 0;
        mfcr2->group_a_immediate_accept = 0;
        mfcr2->group_a_send_calling_party_category_and_switch_to_group_c = R2_SIGA_6;

        mfcr2->group_b_user_busy = R2_SIGB_2;
        mfcr2->group_b_line_free_no_charge = R2_SIGB_5;
        mfcr2->group_b_line_free_charge = R2_SIGB_1;
        /* There does not seem to be a code for unassigned number in Mexico,
           so treat it like busy. This requires that busy is tested before
           unassigned to work properly. */
        mfcr2->group_b_unassigned_number = R2_SIGB_2;

        mfcr2->group_c_send_next_ANI_digit = R2_SIGA_1;
        mfcr2->group_c_repeat_all_DNIS_digits_and_switch_to_group_a = R2_SIGA_2;
        mfcr2->group_c_send_next_DNIS_digit_and_switch_to_group_a = R2_SIGA_5;
        mfcr2->group_c_send_previous_DNIS_digit_and_switch_to_group_a = R2_SIGA_6;
        mfcr2->group_c_switch_to_group_ii = R2_SIGA_3;

        mfcr2->group_b_on_DNIS_timeout = TRUE;
        mfcr2->get_ani_after_dnis = TRUE;

        mfcr2->t3 = 5000;
        /* The time the inbound register has to accept or reject the call (by sending
           the group B tone) is 90s, which is longer than the normal compelled sequence
           timer. */
        break;
    case MFCR2_PROT_NIGERIA:
        mfcr2->max_seize_ack_wait = 10000;
        /* TODO: */
        break;
    case MFCR2_PROT_PANAMA:
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_9;
        break;
    case MFCR2_PROT_PHILIPINNES:
        mfcr2->group_i_end_of_ANI_restricted = R2_SIGI_12;

        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_10;
        break;
    case MFCR2_PROT_ROMANIA:
        /* As per ITU spec. */
        break;
    case MFCR2_PROT_SAUDI_ARABIA:
        mfcr2->group_a_send_calling_party_category = R2_SIGA_5;
        mfcr2->group_b_line_free_charge = R2_SIGB_1;
        /* TODO: */
        break;
    case MFCR2_PROT_SINGAPORE:
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_SUBSCRIBER_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_PRIORITY_SUBSCRIBER_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_MAINTENANCE_CALL] = R2_SIGII_7;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_OPERATOR_CALL] = R2_SIGII_1;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_DATA_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_NATIONAL_PAYPHONE_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_SUBSCRIBER_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_PRIORITY_SUBSCRIBER_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_OPERATOR_CALL] = 0;
        mfcr2->group_ii[UC_CALLER_CATEGORY_INTERNATIONAL_DATA_CALL] = 0;

        mfcr2->group_a_send_previous_DNIS_digit = R2_SIGA_8;
        mfcr2->group_a_send_calling_party_category = R2_SIGA_6;
        mfcr2->group_a_immediate_accept = 0;
        mfcr2->group_a_send_DNIS_digit_n_minus_2 = R2_SIGA_9;
        mfcr2->group_a_send_DNIS_digit_n_minus_3 = 0;
        mfcr2->group_a_repeat_all_DNIS_digits = R2_SIGA_2;
        break;
    case MFCR2_PROT_VENEZUELA:
        mfcr2->group_a_send_next_ANI_digit = R2_SIGA_9;
        mfcr2->group_b_on_DNIS_timeout = TRUE;
        mfcr2->t1 = 45000;
        mfcr2->t2 = 45000;
        mfcr2->t3 = 5000;
        break;
    case MFCR2_PROT_VIETNAM:
        /* As per ITU spec. */
        break;
    case MFCR2_PROT_ITU:
        /* Just leave things alone */
        break;
    }
    /*endswitch*/
}
/*- End of function --------------------------------------------------------*/

static int make_call(uc_t *uc, uc_makecall_t *makecall)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;
    int ch;
    uc_call_t *call;
    const uc_callparms_t *callparms;
    
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Make call\n");

    if (!mfcr2->outgoing_calls_allowed)
        return  UC_RET_BLOCKED;
    /*endif*/

    if (uc->chan[ch].state == UC_STATE_BLOCKED)
        return  UC_RET_BLOCKED;
    /*endif*/
    if (uc->chan[ch].state != UC_STATE_IDLE)
        return  UC_RET_BAD_STATE;
    /*endif*/

    if ((call = uc_createcall(uc, 0)) == NULL)
        return UC_RET_MEMORY_ERROR;
    /*endif*/
    makecall->crn = call->crn;
    callparms = makecall->callparms;

    mfcr2->call = call;
    strcpy(call->callparms.destination_number, callparms->destination_number);
    mfcr2->dnis_len = strlen(call->callparms.destination_number);
    strcpy(call->callparms.originating_number, callparms->originating_number);
    mfcr2->ani_len = strlen(call->callparms.originating_number);
    if (callparms->calling_party_category <= 9)
    {
        /* Fall back to being an ordinary subscriber call, if there is no special code. */
        if (mfcr2->group_ii[callparms->calling_party_category] == 0)
            call->callparms.calling_party_category = UC_CALLER_CATEGORY_NATIONAL_SUBSCRIBER_CALL;
        else
            call->callparms.calling_party_category = callparms->calling_party_category;
        /*endif*/
    }
    else
    {
        /* Just fall back to being an ordinary subscriber call, if the code is bad. */
        call->callparms.calling_party_category = UC_CALLER_CATEGORY_NATIONAL_SUBSCRIBER_CALL;
    }
    /*endif*/
    call->callparms.call_type = UC_CALL_TYPE_UNSPECIFIED;

    call->disconnect_reason = UC_CAUSE_UNSPECIFIED_CAUSE;

    call->incoming_call = FALSE;

    r2_mf_tx_init(&mfcr2->mf_tx_state, TRUE);
    r2_mf_rx_init(&mfcr2->mf_rx_state, FALSE);
    /* We don't need the detector ready until we start sending our first tone.
       Starting it early just makes us more vulnerable to false detection on
       whatever might be in the speech path. */
    call->chan = ch;
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_DIALING;
    /*endif*/
    call->state = UC_STATE_DIALING;
    mfcr2->r2_state = R2_FWD_SEIZE;
    mfcr2->mfc_group = MFC_FWD_STARTING;
    mfcr2->mfc_state = 0;
    if (set_abcd_signal(uc, ch, mfcr2->fwd_abcd_seize))
        return  UC_RET_OK;
    /*endif*/
    mfcr2->seize_ack_timer =
        uc_schedule_event(uc, mfcr2->max_seize_ack_wait, seize_ack_wait_expired, (void *) (intptr_t) ch);

    /* Once we SEIZE we must not clear until at least the SEIZE ACK has been
       received. Failure to do this is a protocol error. If the caller clears
       before SEIZE ACK, we must wait for SEIZE ACK before going to CLEAR FWD. */

    ev.e = UC_EVENT_DIALING;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int request_more_info(uc_t *uc, uc_call_t *call, uc_requestmoreinfo_t *info)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Request more info\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/

    switch (info->type)
    {
    case UC_REQUESTMOREINFO_DESTINATION_NUMBER:
        /* The caller is responding to a UC_EVENT_DIALEDNUMBER, and telling whether to
           get more digits or not. */
        if (uc->chan[ch].state != UC_STATE_DETECTED)
            return  UC_RET_BAD_STATE;
        /*endif*/
        if (info->minimum > UC_MAXPHONENUMLEN)
            return  UC_RET_NOT_AVAILABLE;
        /*endif*/
        if (info->minimum < 0)
        {
            mfcr2->dnis_complete = TRUE;
        }
        else
        {
            if (mfcr2->dnis_len >= info->minimum)
                return  UC_RET_NOT_AVAILABLE;
            /*endif*/
            mfcr2->dnis_complete = FALSE;
        }
        /*endif*/
        if (mfcr2->r2_state != R2_BACK_SEIZE_ACK)
            break;
        /*endif*/
        switch (mfcr2->mfc_state)
        {
        case MFC_SENT_SEIZE_ACK:
            if (mfcr2->dnis_complete  ||  !mfcr2->get_ani_after_dnis)
            {
                /* Get the calling party category, and optionally the ANI, now. If we wait
                   until later there might be problems when the system relies on timeout
                   to terminate the DNIS */
                send_calling_party_category_request(uc);
            }
            else
            {
                mfcr2->mfc_group = MFC_BACK_GROUP_A;
                mfcr2->mfc_state = MFC_SENT_DNIS_REQUEST;
                if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                    break;
                /*endif*/
            }
            /*endif*/
            break;
        case MFC_SENT_DNIS_REQUEST:
            if (mfcr2->dnis_complete)
            {
                /* We have all the DNIS digits we expected. */
                mfcr2->call->callparms.destination_number[mfcr2->dnis_cnt] = '\0';
                if (mfcr2->get_ani_after_dnis)
                {
                    send_calling_party_category_request(uc);
                }
                else
                {
                    mfcr2->mfc_group = MFC_BACK_GROUP_B;
                    mfcr2->mfc_state = MFC_SENT_CHANGE_TO_GROUP_II;
                    if (set_mf_signal(uc, ch, mfcr2->group_a_switch_to_group_ii))
                        break;
                    /*endif*/
                }
                /*endif*/
            }
            else
            {
                if (set_mf_signal(uc, ch, mfcr2->group_a_send_next_DNIS_digit))
                    break;
                /*endif*/
            }
            /*endif*/
            break;
        }
        /*endswitch*/
        return  UC_RET_OK;
    case UC_REQUESTMOREINFO_ORIGINATING_NUMBER:
        break;
    }
    /*endswitch*/
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

static int send_more_info(uc_t *uc, uc_call_t *call, uc_makecall_t *makecall)
{
    uc_log(uc, UC_LOG_FLOW, "Send more info\n");
    /* The makecall structure should now contain more information than when the
       call was initiated (e.g. additional dialed digits). */
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

static int call_ack(uc_t *uc, uc_call_t *call, uc_callack_t *callack)
{
    uc_log(uc, UC_LOG_FLOW, "Call ack\n");
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

static int accept_call(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;

    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Accept call\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (call->state != UC_STATE_OFFERED)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (mfcr2->call->callparms.call_type == UC_CALL_TYPE_NO_CHARGE_CALL)
    {
        mfcr2->mfc_state = MFC_SENT_ACCEPTED_NO_CHARGE;
        if (set_mf_signal(uc, ch, mfcr2->group_b_line_free_no_charge))
            return  UC_RET_BAD_DEVICE;
        /*endif*/
    }
    else
    {
        mfcr2->mfc_state = MFC_SENT_ACCEPTED_CHARGE;
        if (set_mf_signal(uc, ch, mfcr2->group_b_line_free_charge))
            return  UC_RET_BAD_DEVICE;
        /*endif*/
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int answer_call(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Answer call\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    switch (call->state)
    {
    case UC_STATE_OFFERED:
        /* We skipped the accept call step */
        /* Send the signal, indicating line free, charge or free. */
        if (mfcr2->call->callparms.call_type == UC_CALL_TYPE_NO_CHARGE_CALL)
        {
            mfcr2->mfc_state = MFC_SENT_ACCEPTED_NO_CHARGE;
            if (set_mf_signal(uc, ch, mfcr2->group_b_line_free_no_charge))
                return  UC_RET_BAD_DEVICE;
            /*endif*/
        }
        else
        {
            mfcr2->mfc_state = MFC_SENT_ACCEPTED_CHARGE;
            if (set_mf_signal(uc, ch, mfcr2->group_b_line_free_charge))
                return  UC_RET_BAD_DEVICE;
            /*endif*/
        }
        /*endif*/
        break;
    case UC_STATE_ACCEPTED:
        /* Go directly to the answered state */
        if (mfcr2->answer_guard_timer < 0)
        {
            select_active_rxtx(uc, ACTIVE_SPEECH);
            mfcr2->r2_state = R2_BACK_ANSWER;
            if (set_abcd_signal(uc, ch, mfcr2->back_abcd_answer))
                return  UC_RET_OK;
            /*endif*/
            start_connected(uc, mfcr2->call);
        }
        /*endif*/
        call->state = UC_STATE_CONNECTED;
        uc->chan[call->chan].state = UC_STATE_CONNECTED;
        break;
    default:
        return  UC_RET_BAD_STATE;
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int drop_call(uc_t *uc, uc_call_t *call, int cause)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = 0;
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc,
           UC_LOG_FLOW, "Drop call(cause=%s [%d])\n",
           uc_cause2str(cause),
           cause);
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    /* TODO: Check for further state conflicts */
    if (call->state == UC_STATE_NULL
        ||
        call->state == UC_STATE_IDLE
        ||
        call->state == UC_STATE_CALL_DISCONNECTED)
    {
        return  UC_RET_BAD_STATE;
    }
    /*endif*/
    if (call->incoming_call)
    {
        switch (call->state)
        {
        case UC_STATE_OFFERED:
            /* When a call has been offered dropping it requires we give a
               reason first. */
            switch (cause)
            {
            case UC_CAUSE_USER_BUSY:
                mfcr2->mfc_state = MFC_SENT_DEST_BUSY;
                if (set_mf_signal(uc, ch, mfcr2->group_b_user_busy))
                    return  UC_RET_BAD_DEVICE;
                /*endif*/
                break;
            case UC_CAUSE_NETWORK_CONGESTION:
                mfcr2->mfc_state = MFC_SENT_NETWORK_CONGESTION;
                if (set_mf_signal(uc, ch, mfcr2->group_b_network_congestion))
                    return  UC_RET_BAD_DEVICE;
                /*endif*/
                break;
            case UC_CAUSE_UNASSIGNED_NUMBER:
                mfcr2->mfc_state = MFC_SENT_UNASSIGNED_NUMBER;
                if (set_mf_signal(uc, ch, mfcr2->group_b_unassigned_number))
                    return  UC_RET_BAD_DEVICE;
                /*endif*/
                break;
            case UC_CAUSE_DEST_OUT_OF_ORDER:
            default:
                mfcr2->mfc_state = MFC_SENT_DEST_OUT_OF_ORDER;
                if (set_mf_signal(uc, ch, mfcr2->group_b_line_out_of_order))
                    return  UC_RET_BAD_DEVICE;
                /*endif*/
                break;
            }
            /*endswitch*/
            break;
        default:
            if (mfcr2->r2_state == R2_BACK_CLEAR_FWD)
            {
                /* Go to the call disconnected state, but do nothing else.
                   release call will initiate the final release. */
                start_call_disconnected(uc, call);
            }
            else
            {
                mfcr2->r2_state = R2_BACK_CLEAR_BACK;
                if (set_abcd_signal(uc, ch, mfcr2->back_abcd_clear_back))
                    return  UC_RET_OK;
                /*endif*/
                if (mfcr2->play_disconnect_tone)
                {
                    super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_DISCONNECTED]);
                    select_active_rxtx(uc, ACTIVE_SUPER_TONE);
                }
                /*endif*/
            }
            /*endif*/
            break;
        }
        /*endswitch*/
    }
    else
    {
        if (mfcr2->seize_ack_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->seize_ack_timer);
            mfcr2->seize_ack_timer = -1;
        }
        /*endif*/
        if (mfcr2->wait_for_group_b_signal_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->wait_for_group_b_signal_timer);
            mfcr2->wait_for_group_b_signal_timer = -1;
        }
        /*endif*/
        if (mfcr2->await_answer_timer >= 0)
        {
            uc_schedule_del(uc, mfcr2->await_answer_timer);
            mfcr2->await_answer_timer = -1;
        }
        /*endif*/
        switch (call->state)
        {
        case UC_STATE_DIALING:
            if (mfcr2->r2_state == R2_FWD_SEIZE)
            {
                /* This must wait until SEIZE ACK either arrives or we time
                   out before we take any further action with the CAS bits. */
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_SEIZE_ACK;
            }
            else
            {
                if (start_clear_fwd(uc))
                    return  UC_RET_BAD_DEVICE;
                /*endif*/
                mfcr2->r2_state = R2_FWD_CLEAR_FWD_BEFORE_ANSWER;
            }
            /*endif*/
            break;
        case UC_STATE_FAR_DISCONNECTED:
            if (start_clear_fwd(uc))
                return  UC_RET_BAD_DEVICE;
            /*endif*/
            mfcr2->r2_state = R2_FWD_CLEAR_FWD_AFTER_CLEAR_BACK;
            break;
        case UC_STATE_CONNECTED:
        default:
            if (start_clear_fwd(uc))
                return  UC_RET_BAD_DEVICE;
            /*endif*/
            mfcr2->r2_state = R2_FWD_CLEAR_FWD_AFTER_ANSWER;
            if (mfcr2->play_disconnect_tone)
            {
                super_tone_tx_init(&mfcr2->super_tone_tx_state, mfcr2->super_tones->tone[ST_TYPE_DISCONNECTED]);
                select_active_rxtx(uc, ACTIVE_SUPER_TONE);
            }
            /*endif*/
            break;
        }
        /*endswitch*/
    }
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int release_call(uc_t *uc, uc_call_t *call)
{
    mfcr2_signaling_state_t *mfcr2;
    int ch;
    
    ch = 0;    
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Release call\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (call->state != UC_STATE_CALL_DISCONNECTED)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (call->incoming_call)
    {
        select_active_rxtx(uc, ACTIVE_SPEECH);
        mfcr2->r2_state = R2_BACK_RELEASE_GUARD;
        if (set_abcd_signal(uc, ch, mfcr2->abcd_idle))
            return  UC_RET_OK;
        /*endif*/

        /* Ensure a minimum period of release guard for the other end to be 
           sure of seeing it before we can seize for an outgoing call. A seize
           during this timeout will cause the seize logic to prematurely
           terminate this timer, and all will be well. */
        mfcr2->release_guard_timer =
            uc_schedule_event(uc, mfcr2->release_guard_time, release_guard_expired, (void *) call);
    }
    else
    {
        /* There is nothing to do here. The call has completely cleared already. */
        start_idle(uc, call);
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int unblock(uc_t *uc, int ch)
{
    mfcr2_signaling_state_t *mfcr2;
    
    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Unblock\n");
    set_abcd_signal(uc, 0, mfcr2->abcd_idle);
    /* Start a timeout to ensure we do not seize the trunk too agressively.
       That is, before the far end has a chance to recognise the UNBLOCKED
       condition. */
    mfcr2->local_unblocking_timer =
        uc_schedule_event(uc, mfcr2->blocking_release_time, local_unblocking_expired, (void *) (intptr_t) ch);
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int block(uc_t *uc, int ch)
{
    mfcr2_signaling_state_t *mfcr2;
    uc_event_t ev;

    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Block\n");
    mfcr2->local_blocked = TRUE;
    set_abcd_signal(uc, 0, mfcr2->abcd_blocking);
    ev.e = UC_EVENT_LOCALBLOCKED;
    ev.gen.channel = ch;
    ev.gen.crn = 0;
    ev.gen.call = NULL;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int call_control(uc_t *uc, int op, uc_call_t *call, void *data)
{
    uc_log(uc, UC_LOG_FLOW, "Call control(%d)\n", op);
    switch (op)
    {
    case UC_OP_MAKECALL:
        return make_call(uc, (uc_makecall_t *) data);
    case UC_OP_REQUESTMOREINFO:
        return request_more_info(uc, call, (uc_requestmoreinfo_t *) data);
    case UC_OP_SENDMOREINFO:
        return send_more_info(uc, call, (uc_makecall_t *) data);
    case UC_OP_CALLACK:
        return call_ack(uc, call, (uc_callack_t *) data);
    case UC_OP_ACCEPTCALL:
        return accept_call(uc, call);
    case UC_OP_ANSWERCALL:
        return answer_call(uc, call);
    case UC_OP_DROPCALL:
        return drop_call(uc, call, (intptr_t) data);
    case UC_OP_RELEASECALL:
        return release_call(uc, call);
    case UC_OP_UNBLOCK:
        return unblock(uc, (intptr_t) data);
    case UC_OP_BLOCK:
        return block(uc, (intptr_t) data);
    }
    /*endswitch*/
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

static uc_t *create_new(int fd, const char *variant, int mode, int outgoing_calls_allowed)
{
    uc_t *uc;
    mfcr2_signaling_state_t *mfcr2;
    uc_chan_t *chans;
    struct zt_params p;
    struct zt_bufferinfo b;
    int variation;
    int ch;
    int value;
    int len;
    int i;
    int n;
    const char *s;
    char *t;
    
    if ((variation = protocol_variant(variant)) < 0)
    {
        uc_error("Protocol variant '%s' unknown\n", variant);
        return  NULL;
    }
    /*endif*/
    if (ioctl(fd, ZT_GET_PARAMS, &p))
    {
        uc_error("Unable to get channel parameters: %s\n", strerror(errno));
        return  NULL;
    }
    /*endif*/
    if (p.sigtype != ZT_SIG_CAS)
    {
        uc_error("In signaling mode %d, instead of CAS.\n", p.sigtype);
        return  NULL;
    }
    /*endif*/
    if ((uc = malloc(sizeof(*uc))) == NULL)
    {
        uc_error("Failed to allocate buffers.\n");
        return  NULL;
    }
    /*endif*/
    memset(uc, '\0', sizeof(*uc));
    uc->protocol_variant = variation;
    uc->protocol_mode = mode;

    if ((mfcr2 = malloc(sizeof(*mfcr2))) == NULL)
    {
        uc_error("Failed to allocate buffers.\n");
        free(uc);
        return  NULL;
    }
    /*endif*/
    memset(mfcr2, '\0', sizeof(*mfcr2));
    uc->signaling = mfcr2;

    mfcr2->far_unblocking_timer =
    mfcr2->local_unblocking_timer =
    mfcr2->seize_ack_timer =
    mfcr2->release_guard_timer =
    mfcr2->answer_guard_timer =
    mfcr2->await_answer_timer =
    mfcr2->wait_for_group_b_signal_timer =
    mfcr2->clear_back_persistence_timer =
    mfcr2->billing_pulse_timer =
    mfcr2->t1_timer =
    mfcr2->t1a_timer =
    mfcr2->t1b_timer =
    mfcr2->t2_timer =
    mfcr2->t3_timer = -1;

    if ((chans = malloc(sizeof(*chans))) == NULL)
    {
        uc_error("Failed to allocate buffers.\n");
        free(mfcr2);
        free(uc);
        return  NULL;
    }
    /*endif*/
    memset(chans, '\0', sizeof(*chans));
    uc->num_channels = 1;
    uc->chan = chans;
    ch = 0;
    uc->chan[ch].handle = fd;
    uc->chan[ch].state = UC_STATE_BLOCKED;
    uc->chan[ch].logging_level = 0;
    uc->chan[ch].active_rx =
    uc->chan[ch].active_tx = ACTIVE_SPEECH;
    uc->chan[ch].native_law =
    uc->chan[ch].current_law = p.curlaw;
    uc->chan[ch].native_codec =
    uc->chan[ch].api_codec = (p.curlaw == ZT_LAW_ALAW)  ?  UC_CODEC_ALAW  :  (p.curlaw == ZT_LAW_MULAW)  ?  UC_CODEC_ULAW  :  UC_CODEC_UNKNOWN;
#if defined(AUDIO_LOG)
    mfcr2->audio_rx_log =
    mfcr2->audio_tx_log = -1;
#endif

#if defined(AUDIO_LOG)
    {
        char fred[100];
        int yyy;
        static int xxx = 1;

        yyy = xxx++;
        sprintf(fred, "channel-tx-%d.au", yyy);
        mfcr2->audio_tx_log = open(fred, O_CREAT | O_TRUNC | O_RDWR, 0666);
        if (mfcr2->audio_tx_log < 0)
        {
            printf("Failed to open '%s' - %d\n", fred, errno);
            exit(2);
        }
        else
        {
            printf("Opened '%s' - %d\n", fred, mfcr2->audio_tx_log);
        }
        /*endif*/
        sprintf(fred, "channel-rx-%d.au", yyy);
        mfcr2->audio_rx_log = open(fred, O_CREAT | O_TRUNC | O_RDWR, 0666);
        if (mfcr2->audio_rx_log < 0)
        {
            printf("Failed to open '%s' - %d\n", fred, errno);
            exit(2);
        }
        else
        {
            printf("Opened '%s' - %d\n", fred, mfcr2->audio_rx_log);
        }
        /*endif*/
    }
#endif

    if (ioctl(fd, ZT_GET_BUFINFO, &b))
    {
        free(chans);
        free(mfcr2);
        free(uc);
        uc_error("Unable to get channel buffer info: %s\n", strerror(errno));
        return  NULL;
    }
    /*endif*/
    uc->chan[ch].audio_bufsize = b.bufsize;

    load_r2_parameter_set(mfcr2, variation, outgoing_calls_allowed);
    
    s = variant;
    if (variant_element(&s, 1))
    {
        if (sscanf(s, "%d", &value) == 1
            &&
            value >= 0  &&  value <= UC_MAXPHONENUMLEN)
        {
            mfcr2->ani_max_rx_digits = value;
        }
        /*endif*/
    }
    /*endif*/
    s = variant;
    if (variant_element(&s, 2))
    {
        if (sscanf(s, "%d", &value) == 1
            &&
            value >= 0  &&  value <= UC_MAXPHONENUMLEN)
        {
            mfcr2->dnis_max_rx_digits = value;
        }
        /*endif*/
    }
    /*endif*/
    s = variant;
    if (variant_element(&s, 3))
    {
        if (sscanf(s, "%d", &value) == 1)
        {
            mfcr2->play_progress_tones = (value & 1)  ?  TRUE  :  FALSE;
            mfcr2->play_disconnect_tone = (value & 2)  ?  TRUE  :  FALSE;
            mfcr2->play_ringback_tone = (value & 4)  ?  TRUE  :  FALSE;
            mfcr2->get_ani_after_dnis = (value & 8)  ?  TRUE  :  FALSE;
            mfcr2->use_immediate_accept = (value & 0x10)  ?  TRUE  :  FALSE;
            if (mfcr2->group_a_immediate_accept == 0)
                mfcr2->use_immediate_accept = FALSE;
            /*endif*/
        }
        /*endif*/
    }
    /*endif*/

    i = 3;
    while (s = variant, n = variant_element(&s, i))
    {
        if (sscanf(s, "t1=%d", &value) == 1)
        {
            mfcr2->t1 = value;
        }
        else if (sscanf(s, "t1a=%d", &value) == 1)
        {
            mfcr2->t1a = value;
        }
        else if (sscanf(s, "t1b=%d", &value) == 1)
        {
            mfcr2->t1b = value;
        }
        else if (sscanf(s, "t2=%d", &value) == 1)
        {
            mfcr2->t2 = value;
        }
        else if (sscanf(s, "t3=%d", &value) == 1)
        {
            mfcr2->t3 = value;
        }
        else if (sscanf(s, "max-wait-for-group-b-signal=%d", &value) == 1)
        {
            mfcr2->max_wait_for_group_b_signal = value;
        }
        else if (sscanf(s, "max-await-answer=%d", &value) == 1)
        {
            mfcr2->max_await_answer = value;
        }
        else if (sscanf(s, "blocking-release-time=%d", &value) == 1)
        {
            mfcr2->blocking_release_time = value;
        }
        else if (sscanf(s, "answer-guard-time=%d", &value) == 1)
        {
            mfcr2->answer_guard_time = value;
        }
        else if (sscanf(s, "release-guard-time=%d", &value) == 1)
        {
            mfcr2->release_guard_time = value;
        }
        else if (sscanf(s, "max-seize-wait-ack=%d", &value) == 1)
        {
            mfcr2->max_seize_ack_wait = value;
        }
        else if (sscanf(s, "min-clear-back=%d", &value) == 1)
        {
            mfcr2->clear_back_persistence_check = value;
        }
        else if (match_element(s, n, "play-progress-tones") == 0)
        {
            mfcr2->play_progress_tones = TRUE;
        }
        else if (match_element(s, n, "play-disconnect-tone") == 0)
        {
            mfcr2->play_disconnect_tone = TRUE;
        }
        else if (match_element(s, n, "play-ringback-tone") == 0)
        {
            mfcr2->play_ringback_tone = TRUE;
        }
        else if (match_element(s, n, "get-ani-after-dnis") == 0)
        {
            mfcr2->get_ani_after_dnis = TRUE;
        }
        else if (match_element(s, n, "use-immediate-accept") == 0)
        {
            if (mfcr2->group_a_immediate_accept != 0)
                mfcr2->use_immediate_accept = TRUE;
            /*endif*/
        }
        else if (match_element(s, n, "progressive-dnis") == 0)
        {
            mfcr2->use_dialed_number_events = TRUE;
        }
        /*endif*/
        i++;
    }
    /*endwhile*/
    
    if (mfcr2->play_progress_tones  ||  mfcr2->play_disconnect_tone  ||  mfcr2->play_ringback_tone)
    {
        s = variant;
        len = variant_element(&s, 0);
        t = malloc(len + 1);
        strncpy(t, s, len);
        t[len] = '\0';
        mfcr2->super_tones = get_supervisory_tone_set(t);
        free(t);
        if (mfcr2->super_tones == NULL)
        {
            free(chans);
            free(mfcr2);
            free(uc);
            uc_error("Failed to load supervisory tones.\n");
            return  NULL;
        }
        /*endif*/
    }
    /*endif*/

    hash_init_table(&uc->calls, HASH_ONE_WORD_KEYS);

    mfcr2->current_abcd_in = 0xFF;
    mfcr2->r2_state = R2_IDLE;
    mfcr2->mfc_group = 0;
    mfcr2->mfc_state = 0;
    if (set_abcd_signal(uc, ch, mfcr2->abcd_blocking) == UC_RET_OK)
        abcd_signaling_event(uc, ch);
    /*endif*/
    return uc;
}
/*- End of function --------------------------------------------------------*/

static int delete_context(uc_t *uc)
{
    #if defined(AUDIO_LOG)
    close(mfcr2->audio_rx_log);
    close(mfcr2->audio_tx_log);
    #endif
    hash_delete_table(&uc->calls);
    free(uc->chan);
    free(uc->signaling);
    free(uc);
}
/*- End of function --------------------------------------------------------*/

static int channel_open(const char *variant, char *dev)
{
    struct zt_bufferinfo bi;
    struct zt_gains g;
    int x;
    int fd;
    int i;
    int chan;

    if ((chan = atoi(dev)) > 0)
    {
        if ((fd = open("/dev/zap/channel", O_RDWR | O_NONBLOCK)) < 0)
            return -1;
        /*endif*/
        if (ioctl(fd, ZT_SPECIFY, &chan))
        {
            x = errno;
            close(fd);
            errno = x;
            return -1;
        }
        /*endif*/
    }
    else
    {
        if ((fd = open(dev, O_RDWR | O_NONBLOCK)) < 0)
            return -1;
        /*endif*/
    }
    /*endif*/
    if (ioctl(fd, ZT_GET_BUFINFO, &bi) < 0)
    {
        x = errno;
        close(fd);
        errno = x;
        return -1;
    }
    /*endif*/
    bi.txbufpolicy = ZT_POLICY_IMMEDIATE;
    bi.rxbufpolicy = ZT_POLICY_IMMEDIATE;
    bi.numbufs = 4;
    bi.bufsize = 160;
    if (ioctl(fd, ZT_SET_BUFINFO, &bi) < 0)
    {
        x = errno;
        close(fd);
        errno = x;
        return -1;
    }
    /*endif*/
    if (ioctl(fd, ZT_CHANNO, &chan))
    {
        x = errno;
        close(fd);
        errno = x;
        return -1;
    }
    /*endif*/
    /* Set default gains */
    g.chan = 0;
    for (i = 0;  i < 256;  i++)
    {
        g.rxgain[i] = i;
        g.txgain[i] = i;
    }
    /*endfor*/
    if (ioctl(fd, ZT_SETGAINS, &g) < 0)
    {
        x = errno;
        close(fd);
        errno = x;
        return -1;
    }
    /*endif*/
    return fd;
}
/*- End of function --------------------------------------------------------*/

static int channel_close(int fd)
{
    close(fd);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static char *error_message(uc_t *uc, int code)
{
    switch (code)
    {
    case MFCR2_PROTOCOL_FAIL_T1_TIMEOUT:
        return "T1 timed out";
    case MFCR2_PROTOCOL_FAIL_T2_TIMEOUT:
        return "T2 timed out";
    case MFCR2_PROTOCOL_FAIL_T3_TIMEOUT:
        return "T3 timed out";
    case MFCR2_PROTOCOL_FAIL_UNEXPECTED_MF_SIGNAL:
        return "Unexpected MF6 signal";
    case MFCR2_PROTOCOL_FAIL_UNEXPECTED_CAS:
        return "Unexpected CAS bit pattern";
    case MFCR2_PROTOCOL_FAIL_INVALID_STATE:
        return "Invalid state";
    case MFCR2_PROTOCOL_FAIL_SET_CAS_FAILURE:
        return "Failed to set CAS signaling bits";
    case MFCR2_PROTOCOL_FAIL_SEIZE_ACK_TIMEOUT:
        return "Seize ack timed out";
    case MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR:
        return "Device I/O error";
    }
    /*endswitch*/
    return "Undefined";
}
/*- End of function --------------------------------------------------------*/

static int get_device_handles(uc_t *uc, int ch, int *fe, int *be)
{
    if (fe)
        *fe = uc->chan[ch].handle;
    /*endif*/
    if (be)
        *be = uc->chan[ch].handle;
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int channel_set_api_codec(uc_t *uc, int ch, int codec)
{
    int linear;
    int law;

    //uc_log(uc, UC_LOG_FLOW, "Channel set API codec - %d\n", codec);
    switch (codec)
    {
    case UC_CODEC_DEFAULT:
        linear = 0;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = uc->chan[ch].native_law;
        if (law == ZT_LAW_ALAW)
            codec = UC_CODEC_ALAW;
        else if (law == ZT_LAW_MULAW)
            codec = UC_CODEC_ULAW;
        /*endif*/
        break;
    case UC_CODEC_LINEAR16:
        linear = 1;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = uc->chan[ch].native_law;
        break;
    case UC_CODEC_ALAW:
        linear = 0;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = ZT_LAW_ALAW;
        break;
    case UC_CODEC_ULAW:
        linear = 0;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = ZT_LAW_MULAW;
        break;
    default:
        return  UC_RET_UNSUPPORTED;
    }
    /*endswitch*/
    if (uc->chan[ch].current_law != law)
    {
        uc->chan[ch].current_law = law;
        if (ioctl(uc->chan[ch].handle, ZT_SETLAW, &law))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
    }
    /*endif*/
    uc->chan[ch].api_codec = codec;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int channel_write(uc_t *uc, int ch, uint8_t *buf, int len)
{
    int xlen;
    int i;

    if (uc->chan[ch].active_tx != ACTIVE_SPEECH)
        return len;
    /*endif*/
    if (uc->chan[ch].channel_write_codec)
        len = uc->chan[ch].channel_write_codec(uc, ch, uc->chan[ch].channel_write_codec_user_data, buf, len);
    /*endif*/
    /* NASTY BODGE ALERT:
       EAGAIN means many things, including write buffer full. We have no way to
       know what actually caused it. We need to limit the number of write tries,
       otherwise buffer full can foul up the whole machine with a storm of
       retries. Sleeps between tries are possible, but that is not a cure.
       If we get a buffer full condition, something is writing too fast.
       Loosing some data is the only reasonable strategy. */
    for (i = 0;  i < 5;  i++)
    {
        xlen = write(uc->chan[ch].handle, buf, len);
        if (xlen >= len)
            break;
        /*endif*/
        if (xlen < 0)
        {
            if (errno == ELAST)
            {
                check_event(uc);
                continue;
            }
            /*endif*/
            if (errno == EAGAIN)
                continue;
            /*endif*/
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            break;
        }
        /*endif*/
        len -= xlen;
        buf += len;
    }
    /*endfor*/
    if (errno == EAGAIN)
        xlen = 1;
    /*endif*/
    return  xlen;
}
/*- End of function --------------------------------------------------------*/

static int channel_flush(uc_t *uc, int ch, int which)
{
    mfcr2_signaling_state_t *mfcr2;
    int x;

    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel flush\n");
    if ((which & 1))
    {
        x = ZT_FLUSH_WRITE;
        if (ioctl(uc->chan[ch].handle, ZT_FLUSH, &x))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
    }
    /*endif*/
    if ((which & 2))
    {
        /* TODO: implement input flush */
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int channel_gains(uc_t *uc, int ch, float rxgain, float txgain)
{
    mfcr2_signaling_state_t *mfcr2;
    struct zt_gains g;
    float gain;
    int j;
    int k;

    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel gains\n");
    g.chan = 0;
    if (rxgain != 0.0)
    {
        /* Calculate linear value of rx gain */
        gain = pow(10.0, rxgain/20.0);
        if (uc->chan[ch].current_law == UC_CODEC_ALAW)
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) alaw_to_linear(j))*gain);
                g.rxgain[j] = linear_to_alaw(k);
            }
            /*endfor*/
        }
        else
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) ulaw_to_linear(j))*gain);
                g.rxgain[j] = linear_to_ulaw(k);
            }
            /*endfor*/
        }
        /*endif*/
    }
    else
    {
        for (j = 0;  j < 256;  j++)
            g.rxgain[j] = j;
        /*endfor*/
    }
    /*endif*/
    if (txgain != 0.0)
    {
        /* Calculate linear value of tx gain */
        gain = pow(10.0, txgain/20.0);
        if (uc->chan[ch].current_law == UC_CODEC_ALAW)
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) alaw_to_linear(j))*gain);
                g.txgain[j] = linear_to_alaw(k);
            }
            /*endfor*/
        }
        else
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) ulaw_to_linear(j))*gain);
                g.txgain[j] = linear_to_ulaw(k);
            }
            /*endfor*/
        }
        /*endif*/
    }
    else
    {
        for (j = 0;  j < 256;  j++)
            g.txgain[j] = j;
        /*endfor*/
    }
    /*endif*/
    if (ioctl(uc->chan[ch].handle, ZT_SETGAINS, &g))
    {
        if (uc->chan[ch].channel_error)
            uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
        /*endif*/
        return  UC_RET_DEVICE_ERROR;
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int channel_echo_cancel(uc_t *uc, int ch, int op)
{
    mfcr2_signaling_state_t *mfcr2;
    int x;

    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel echo cancel\n");
    switch (op)
    {
    case UC_ECHO_CANCEL_OFF:
    case UC_ECHO_CANCEL_ON:
        x = op;
        if (ioctl(uc->chan[ch].handle, ZT_ECHOCANCEL, &x) < 0)
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    case UC_ECHO_CANCEL_NOTRAINING:
    case UC_ECHO_CANCEL_TRAINING:
        x = op - UC_ECHO_CANCEL_NOTRAINING;
        if (ioctl(uc->chan[ch].handle, ZT_ECHOTRAIN, &x) < 0)
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    default:
        return UC_RET_BAD_PARAMETER;
    }
    /*endswitch*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int channel_switching(uc_t *uc, int ch, int op, int dest, int parms)
{
    struct zt_confinfo zi;
    mfcr2_signaling_state_t *mfcr2;
    int x;

    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel switching\n");
    switch (op)
    {
    case UC_SWITCHING_FREE:
        memset(&zi, 0, sizeof(zi));
        if (ioctl(uc->chan[ch].handle, ZT_SETCONF, &zi))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    case UC_SWITCHING_UNMUTE:
    case UC_SWITCHING_MUTE:
        x = op - UC_SWITCHING_UNMUTE;
        if (ioctl(uc->chan[ch].handle, ZT_CONFMUTE, &x))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    case UC_SWITCHING_CONNECT:
        memset(&zi, 0, sizeof(zi));
        zi.confno = dest;
        zi.confmode = parms;
        if (ioctl(uc->chan[ch].handle, ZT_SETCONF, &zi))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    default:
        return UC_RET_BAD_PARAMETER;
    }
    /*endswitch*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int abcd_control(uc_t *uc, int ch, int abcd)
{
    int x;

    x = abcd;
    if (ioctl(uc->chan[ch].handle, ZT_SETTXBITS, &x))
    {
        if (errno != EINPROGRESS)
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return  UC_RET_DEVICE_ERROR;
        }
        /*endif*/
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int check_abcd(uc_t *uc, int ch, int *abcd)
{
    int x;

    if (ioctl(uc->chan[ch].handle, ZT_GETRXBITS, &x))
    {
        if (uc->chan[ch].channel_error)
            uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
        /*endif*/
        return  UC_RET_DEVICE_ERROR;
    }
    /*endif*/
    *abcd = x;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int check_alarms(uc_t *uc, int ch, int *alarms)
{
    ZT_SPANINFO zi;

    memset(&zi, 0, sizeof(zi));
    zi.spanno = 0;
    if (ioctl(uc->chan[ch].handle, ZT_SPANSTAT, &zi))
    {
        uc_error("Unable to get alarm status: %s\n", strerror(errno));
        if (uc->chan[ch].channel_error)
            uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, MFCR2_PROTOCOL_FAIL_DEVICE_IO_ERROR);
        /*endif*/
        return  UC_RET_DEVICE_ERROR;
    }
    /*endif*/
    *alarms = zi.alarms;
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
